/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package malik.emulator.media.sound;

import malik.emulator.application.*;
import malik.emulator.util.*;

public final class SampledSoundPlayer extends SoundPlayer
{
    public static final int MAX_CHANNELS = 256;
    public static final int MAX_SAMPLES_PER_SECOND = 48000;
    public static final int MAX_BLOCK_LENGTH = 0x00010000;
    public static final int MIN_BLOCK_LENGTH = 0x00000100;

    public static SampledSoundPlayer open(int channels, int samplesPerSecond, int blockLength) throws SoundPlayerException {
        int handle;
        if(channels < 1 || channels > MAX_CHANNELS)
        {
            throw new IllegalArgumentException("SampledSoundPlayer.open: аргумент channels имеет недопустимое значение.");
        }
        if(samplesPerSecond < 1 || samplesPerSecond > MAX_SAMPLES_PER_SECOND)
        {
            throw new IllegalArgumentException("SampledSoundPlayer.open: аргумент samplesPerSecond имеет недопустимое значение.");
        }
        if(blockLength < MIN_BLOCK_LENGTH || blockLength > MAX_BLOCK_LENGTH)
        {
            throw new IllegalArgumentException("SampledSoundPlayer.open: аргумент blockLength имеет недопустимое значение.");
        }
        if((handle = (int) MalikSystem.syscall(samplesPerSecond & 0xffff | 16 << 16 | (channels - 1) << 24, blockLength, 0x0030)) == 0)
        {
            throw new SoundPlayerException((new StringBuilder()).
                append("SampledSoundPlayer.open: системе не хватает памяти на создание нового проигрывателя или возможности системы не позволяют проигрывать ИКМ с параметрами: ").
                append(samplesPerSecond).append(" Гц, ").append(channels == 1 ? "моно." : channels == 2 ? "стерео." : Integer.toString(channels).concat(" каналов.")).
            toString());
        }
        return new SampledSoundPlayer(handle);
    }

    int handle;
    private final DataDescriptor descriptor;

    private SampledSoundPlayer(int handle) {
        this.handle = handle;
        this.descriptor = new DataDescriptor();
        Run.instance.setPCMPlayerListener(handle, new PCMPlayerListener() {
            public void endOfBlock(int playerHandle, int blockIndexForLoad) {
                SampledSoundPlayer parent;
                SoundPlayerListener listener;
                if(playerHandle == (parent = SampledSoundPlayer.this).handle && (listener = parent.listener) != null) listener.endOfBlock(parent, blockIndexForLoad);
            }
        });
    }

    public void close() {
        synchronized(descriptor)
        {
            int h;
            if((h = handle) != 0)
            {
                handle = 0;
                listener = null;
                MalikSystem.syscall((long) h, 0x0037);
                Run.instance.setPCMPlayerListener(h, null);
            }
        }
    }

    public int loadBlock(long[] packedSamples, int offset, int length) throws SoundPlayerException {
        int error;
        int result;
        DataDescriptor d;
        if(packedSamples == null)
        {
            throw new NullPointerException("SampledSoundPlayer.loadBlock: аргумент packedSamples равен нулевой ссылке.");
        }
        Array.checkBound("SampledSoundPlayer.loadBlock", packedSamples.length, offset, length);
        if(length <= 0) return 0;
        error = 0;
        synchronized(d = descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                d.setDataInfo(Array.getFirstElementAddress(packedSamples) + (offset << 3), length);
                result = (int) MalikSystem.syscall(h, d.getDescriptorAddress(), 0x0031);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.loadBlock: проигрыватель закрыт.");
        }
        return result;
    }

    public int reset() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 0, 0x0032);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.reset: проигрыватель закрыт.");
        }
        return result;
    }

    public int start() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 1, 0x0032);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.start: проигрыватель закрыт.");
        }
        return result;
    }

    public int stop() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 2, 0x0032);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.stop: проигрыватель закрыт.");
        }
        return result;
    }

    public int setVolume(int volume) throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, super.setVolume(volume), 0x0036);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.setVolume: проигрыватель закрыт.");
        }
        return result;
    }

    public int setCurrentBlockPosition(int position) throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, position, 0x0034);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.setCurrentBlockPosition: проигрыватель закрыт.");
        }
        return result;
    }

    public int getState() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall((long) h, 0x0033);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.getState: проигрыватель закрыт.");
        }
        return result;
    }

    public int getVolume() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, -1, 0x0036);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.getVolume: проигрыватель закрыт.");
        }
        return result;
    }

    public int getCurrentBlockIndex() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) (MalikSystem.syscall(h, 0, 0x0035) >> 32);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.getCurrentBlockIndex: проигрыватель закрыт.");
        }
        return result;
    }

    public int getCurrentBlockPosition() throws SoundPlayerException {
        int result;
        int error = 0;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                result = (int) MalikSystem.syscall(h, 0, 0x0035);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.getCurrentBlockPosition: проигрыватель закрыт.");
        }
        return result;
    }

    public long getCurrentBlockIndexAndPosition() throws SoundPlayerException {
        int error = 0;
        long result;
        synchronized(descriptor)
        {
            label0:
            {
                int h;
                if((h = handle) == 0)
                {
                    error = 1;
                    result = 0L;
                    break label0;
                }
                result = MalikSystem.syscall(h, 0, 0x0035);
            }
        }
        if(error == 1)
        {
            throw new SoundPlayerException("SampledSoundPlayer.getCurrentBlockIndexAndPosition: проигрыватель закрыт.");
        }
        return result;
    }
}
