/*
	Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
	и других спецификаций для функционирования компактных приложений на языке
	Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

	Copyright © 2016, 2019 Малик Разработчик

	Это свободная программа: вы можете перераспространять ее и/или изменять
	ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
	в каком она была опубликована Фондом свободного программного обеспечения;
	либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

	Эта программа распространяется в надежде, что она будет полезной,
	но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
	или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
	общественной лицензии GNU.

	Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
	вместе с этой программой. Если это не так, см.
	<https://www.gnu.org/licenses/>.
*/


package malik.emulator.media.sound;

import malik.emulator.application.*;
import malik.emulator.util.*;

public final class SoundPlayerSampled extends Object
		implements PlayerPCMLoadBlockHandler, SoundPlayer
{
	public static final int MIN_BLOCK_LENGTH = 0x00000100;
	public static final int MAX_BLOCK_LENGTH = 0x00010000;

	public static SoundPlayerSampled open(int samplesPerSecond,
			int bitsPerSample, int channelsCount, int blockLength)
			throws SoundPlayerException
	{
		int handle;
		if((handle = (int) MalikSystem.syscall(samplesPerSecond | (bitsPerSample << 16) |
				((channelsCount - 1) << 24), blockLength, 0x0030)) == 0)
		{
			throw new SoundPlayerException("SoundPlayerSampled.open: " +
					"системе не хватает памяти на создание нового PCM-проигрывателя или " +
					"возможности системы не позволяют воспроизводить звук в ИКМ с параметрами " +
					samplesPerSecond + " Гц, " + bitsPerSample + " бит, " +
					(channelsCount == 1 ? "моно." : (channelsCount == 2 ? "стерео." :
					Integer.toString(channelsCount) + " каналов.")));
		}
		return new SoundPlayerSampled(handle);
	}


	private int handle;
	private DataDescriptor descriptor;
	private SoundPlayerListener listener;
	private Object lock;

	private SoundPlayerSampled(int handle)
	{
		this.handle = handle;
		this.descriptor = new DataDescriptor();
		this.lock = new Object();
		Run.getInstance().setPlayerPCMLoadBlockHandler(handle, this);
	}

	public void loadPCMBlock(int playerPCMHandle, int blockIndex)
	{
		SoundPlayerListener listener;
		if((listener = this.listener) != null && playerPCMHandle == this.handle)
		{
			listener.loadBlock(this, blockIndex);
		}
	}

	public void setSoundPlayerListener(SoundPlayerListener listener)
	{
		this.listener = listener;
	}

	public void close()
	{
		int h;
		synchronized(lock)
		{
			if((h = handle) != 0)
			{
				handle = 0;
				MalikSystem.syscall((long) h, 0x0037);
				Run.getInstance().setPlayerPCMLoadBlockHandler(h, null);
			}
		}
	}

	public int loadBlock(long[] packedSamples, int offset, int length)
	{
		int h;
		int lim;
		int len;
		int result;
		DataDescriptor d;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		if(packedSamples == null)
		{
			throw new NullPointerException("SoundPlayerSampled.loadBlock: " +
					"параметр packedSamples равен нулевой ссылке.");
		}
		if((lim = offset + length) > (len = packedSamples.length) ||
				lim < offset || offset > len || offset < 0)
		{
			throw new ArrayIndexOutOfBoundsException("SoundPlayerSampled.loadBlock: " +
					"индекс выходит из диапазона.");
		}
		synchronized(lock)
		{
			(d = descriptor).setDataInfo(Array.getFirstElementAddress(packedSamples) +
					(offset << 3), length);
			result = (int) MalikSystem.syscall(h, d.getDescriptorAddress(), 0x0031);
		}
		return result;
	}

	public int reset()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x0032);
	}

	public int start()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 1, 0x0032);
	}

	public int stop()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 2, 0x0032);
	}

	public int setVolume(int volume)
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h,
				volume < 0 ? 0 : (volume > 0x7f ? 0x7f : volume), 0x0036);
	}

	public int setCurrentBlockPosition(int position)
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, position, 0x0034);
	}

	public int getState()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x0033);
	}

	public int getVolume()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, -1, 0x0036);
	}

	public int getCurrentBlockPosition()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) MalikSystem.syscall(h, 0, 0x0035);
	}

	public int getCurrentBlockIndex()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return (int) (MalikSystem.syscall(h, 0, 0x0035) >> 32);
	}

	public long getCurrentBlockAndPosition()
	{
		int h;
		if((h = handle) == 0)
		{
			throw new IllegalStateException("SoundPlayer: " +
					"использование после вызова метода close().");
		}
		return MalikSystem.syscall(h, 0, 0x0035);
	}
}
