/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package malik.emulator.microedition.system.protocol.sms;

import java.io.*;
import javax.wireless.messaging.*;
import malik.emulator.microedition.io.*;
import malik.emulator.microedition.system.protocol.*;

public class ClientMessageConnection extends CustomConnection implements MessageConnection
{
    public static final int MAXIMUM_LENGTH = 0x0400;
    public static final int LOCAL_PORT = 1200;
    public static final String LOCAL_ADDRESS = "sms://:" + LOCAL_PORT;

    private static int getPort(String address) {
        int result;
        if(address == null || !address.toLowerCase().startsWith("sms://:")) return LOCAL_PORT;
        try
        {
            result = Integer.parseInt(address.substring("sms://:".length()));
        }
        catch(NumberFormatException e)
        {
            return LOCAL_PORT;
        }
        return result < 0x0000 || result > 0xffff ? LOCAL_PORT : result;
    }

    public ClientMessageConnection(String remoteAddress) {
        super(remoteAddress);
    }

    public void setMessageListener(MessageListener listener) throws IOException {
        if(isConnectionClosed())
        {
            throw new ConnectionClosedException("MessageConnection.setMessageListener: соединение закрыто.");
        }
        throw new IOException("MessageConnection.setMessageListener: клиентское соединение не может принимать сообщения.");
    }

    public void send(Message message) throws IOException {
        int len;
        ShortMessage ourmsg;
        if(isConnectionClosed())
        {
            throw new ConnectionClosedException("MessageConnection.send: соединение закрыто.");
        }
        if(message == null)
        {
            throw new NullPointerException("MessageConnection.send: аргумент message равен нулевой ссылке.");
        }
        if(!(message instanceof ShortMessage) || (ourmsg = (ShortMessage) message).creator != this)
        {
            throw new IllegalArgumentException("MessageConnection.send: аргумент message не был создан методом newMessage.");
        }
        if((len = ourmsg.getLength()) < 0 || len > MAXIMUM_LENGTH)
        {
            throw new IllegalArgumentException("MessageConnection.send: аргумент message имеет большую длину.");
        }
        InboundConnections.instance.notifyReceiveData(InboundConnections.UDP, getPort(ourmsg.recipientAddress), ourmsg.createInboundCopy());
    }

    public int numberOfSegments(Message message) {
        int len;
        ShortMessage ourmsg;
        return !(message instanceof ShortMessage) || (ourmsg = (ShortMessage) message).creator != this || (len = ourmsg.getLength()) < 0 || len > MAXIMUM_LENGTH ? 0 : 1;
    }

    public Message receive() throws IOException {
        if(isConnectionClosed())
        {
            throw new ConnectionClosedException("MessageConnection.receive: соединение закрыто.");
        }
        throw new IOException("MessageConnection.receive: клиентское соединение не может принимать сообщения.");
    }

    public Message newMessage(String type) {
        if(BINARY_MESSAGE.equals(type)) return new BinaryShortMessage(this, false, LOCAL_ADDRESS, super.getURL(), ShortMessage.EMPTY_TIMESTAMP, null);
        if(TEXT_MESSAGE.equals(type)) return new TextShortMessage(this, false, LOCAL_ADDRESS, super.getURL(), ShortMessage.EMPTY_TIMESTAMP, null);
        throw new IllegalArgumentException("MessageConnection.newMessage: аргумент type имеет недопустимое значение.");
    }

    public Message newMessage(String type, String address) {
        if(BINARY_MESSAGE.equals(type)) return new BinaryShortMessage(this, false, LOCAL_ADDRESS, address, ShortMessage.EMPTY_TIMESTAMP, null);
        if(TEXT_MESSAGE.equals(type)) return new TextShortMessage(this, false, LOCAL_ADDRESS, address, ShortMessage.EMPTY_TIMESTAMP, null);
        throw new IllegalArgumentException("MessageConnection.newMessage: аргумент type имеет недопустимое значение.");
    }

    public final void setURL(String url) {
        throw new IllegalStateException("CustomConnection.setURL: невозможно изменить адрес соединения.");
    }
}
