/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package java.io;

public class ByteArrayInputStream extends InputStream
{
    protected int pos;
    protected int mark;
    protected int count;
    protected byte[] buf;
    private final Object monitor;

    public ByteArrayInputStream(byte[] buffer) {
        this.count = buffer == null ? 0 : buffer.length;
        this.buf = buffer;
        this.monitor = new Object();
    }

    public ByteArrayInputStream(byte[] buffer, int offset, int length) {
        long bounds;
        offset = (int) (bounds = Array.intersectBound(buffer == null ? 0 : buffer.length, offset, length));
        length = (int) (bounds >> 32);
        this.pos = offset;
        this.count = offset + length;
        this.buf = buffer;
        this.monitor = new Object();
    }

    public void close() throws IOException {
    }

    public void reset() {
        synchronized(monitor)
        {
            pos = mark;
        }
    }

    public void mark(int readLimit) {
        synchronized(monitor)
        {
            mark = pos;
        }
    }

    public boolean markSupported() {
        return true;
    }

    public int available() {
        int result;
        synchronized(monitor)
        {
            result = count - pos;
        }
        return result;
    }

    public int read() {
        int result;
        synchronized(monitor)
        {
            int p;
            if((p = pos) >= count)
            {
                result = -1;
            } else
            {
                result = buf[p++] & 0xff;
                pos = p;
            }
        }
        return result;
    }

    public int read(byte[] dst, int offset, int length) {
        int result;
        if(dst == null)
        {
            throw new NullPointerException("ByteArrayInputStream.read: аргумент dst равен нулевой ссылке.");
        }
        Array.checkBound("ByteArrayInputStream.read", dst.length, offset, length);
        synchronized(monitor)
        {
            int p;
            int remaining;
            if((remaining = count - (p = pos)) <= 0)
            {
                result = -1;
            } else
            {
                result = length <= remaining ? length : remaining;
                Array.copy(buf, p, dst, offset, result);
                pos = p + result;
            }
        }
        return result;
    }

    public long skip(long quantity) {
        long result;
        if(quantity <= 0L) return 0L;
        synchronized(monitor)
        {
            int p;
            long remaining = (long) (count - (p = pos));
            result = quantity <= remaining ? quantity : remaining;
            pos = p + (int) result;
        }
        return result;
    }
}
