{
    PixelGraphicHighQualityScaling реализует алгоритм масштабирования растровой
    графики «высококачественное масштабирование», который написал Максим Степин.

    Copyright © 2003 Максим Степин
    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять её и/или
    изменять её на условиях Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 2 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она может быть полезна,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЁННЫХ ЦЕЛЕЙ. Подробнее см. в Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <http://www.gnu.org/licenses/>.
}

unit PixelGraphicHighQualityScaling;

{$MODE DELPHI}

interface

uses
    Lang,
    IntelMMX,
    PixelGraphicScaling;

{%region public }
type
    Algorithm = class(RefCountInterfacedObject, PixelGraphicScalingAlgorithm)
    private
        pixels16bit: short_Array1d;
        inp: PIntArray;
        width: int;
        height: int;
        inScanline: int;
        square3x3w: int_Array1d;
        square3x3c: int_Array1d;
    public
        constructor create();
        procedure setSource(pixels: PIntArray; width, height, scanline: int);
        procedure scale2x(outp: PIntArray; outScanline: int; turned: boolean);
        procedure scale3x(outp: PIntArray; outScanline: int; turned: boolean);
        procedure scale4x(outp: PIntArray; outScanline: int; turned: boolean);
        function getSourceWidth(): int;
        function getSourceHeight(): int;
    end;
{%endregion}

implementation

{%region private }
var
    RGBTOYUV: int_Array1d;
    REGBLANK: long = $0000000000000000;
    TRESHOLD: long = $0000000000300706;
    CONST3  : long = $0003000300030003;
    CONST5  : long = $0005000500050005;
    CONST6  : long = $0006000600060006;
    CONST7  : long = $0007000700070007;
    CONST14 : long = $000e000e000e000e;
{%endregion}

{%region routine }
    procedure init();
    var
        index: int;
        i: int;
        j: int;
        k: int;
        r: int;
        g: int;
        b: int;
        y: int;
        u: int;
        v: int;
    begin
        RGBTOYUV := int_Array1d_create($10000);
        index := 0;
        for i := 0 to 31 do for j := 0 to 63 do for k := 0 to 31 do begin
            r := i shl 3;
            g := j shl 2;
            b := k shl 3;
            y := (r + g + b) shr 2;
            u := 128 + sar(r - b, 2);
            v := 128 + sar(-r + 2 * g - b, 3);
            RGBTOYUV[index] := (y shl 16) + (u shl 8) + v;
            inc(index);
        end;
    end;

    function diff(w1, w2: int): int; assembler; nostackframe;
    asm
                push    ebx
                lea     ebx, [eax+$00]
                xor     eax, eax
                cmp     ebx, edx
                je      @0
                mov     ecx, [RGBTOYUV]
                movd    mm1, [ecx+ebx*4]
                movd    mm2, [ecx+edx*4]
                movq    mm3, mm1
                psubusb mm1, mm2
                psubusb mm2, mm3
                por     mm1, mm2
                psubusb mm1, [TRESHOLD]
                movd    eax, mm1
        @0:     pop     ebx
    end;

    function interp1(c1, c2: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 3 + c2) div 4; }
                movd    mm1, eax
                movd    mm2, edx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                pmullw  mm1, [CONST3]
                paddw   mm1, mm2
                psrlw   mm1, $02
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp2(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 2 + c2 + c3) div 4; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                psllw   mm1, $01
                paddw   mm1, mm2
                paddw   mm1, mm3
                psrlw   mm1, $02
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp3(c1, c2: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 7 + c2) div 8; }
                movd    mm1, eax
                movd    mm2, edx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                pmullw  mm1, [CONST7]
                paddw   mm1, mm2
                psrlw   mm1, $03
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp4(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 2 + (c2 + c3) * 7) div 16; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                psllw   mm1, $01
                paddw   mm2, mm3
                pmullw  mm2, [CONST7]
                paddw   mm1, mm2
                psrlw   mm1, $04
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp5(c1, c2: int): int; assembler; nostackframe;
    asm
        { result := (c1 + c2) div 2; }
                movd    mm1, eax
                movd    mm2, edx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                paddw   mm1, mm2
                psrlw   mm1, $01
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp6(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 5 + c2 * 2 + c3) div 8; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                pmullw  mm1, [CONST5]
                psllw   mm2, $01
                paddw   mm1, mm2
                paddw   mm1, mm3
                psrlw   mm1, $03
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp7(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 6 + c2 + c3) div 8; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                pmullw  mm1, [CONST6]
                paddw   mm1, mm2
                paddw   mm1, mm3
                psrlw   mm1, $03
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp8(c1, c2: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 5 + c2 * 3) div 8; }
                movd    mm1, eax
                movd    mm2, edx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                pmullw  mm1, [CONST5]
                pmullw  mm2, [CONST3]
                paddw   mm1, mm2
                psrlw   mm1, $03
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp9(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 2 + (c2 + c3) * 3) div 8; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                psllw   mm1, $01
                paddw   mm2, mm3
                pmullw  mm2, [CONST3]
                paddw   mm1, mm2
                psrlw   mm1, $03
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;

    function interp10(c1, c2, c3: int): int; assembler; nostackframe;
    asm
        { result := (c1 * 14 + c2 + c3) div 16; }
                movd    mm1, eax
                movd    mm2, edx
                movd    mm3, ecx
                punpcklbw mm1, [REGBLANK]
                punpcklbw mm2, [REGBLANK]
                punpcklbw mm3, [REGBLANK]
                pmullw  mm1, [CONST14]
                paddw   mm1, mm2
                paddw   mm1, mm3
                psrlw   mm1, $04
                packuswb mm1, [REGBLANK]
                movd    eax, mm1
    end;
{%endregion}

{%region Algorithm }
    constructor Algorithm.create();
    begin
        inherited create();
        square3x3w := int_Array1d_create(3 * 3);
        square3x3c := int_Array1d_create(3 * 3);
    end;

    procedure Algorithm.setSource(pixels: PIntArray; width, height, scanline: int);
    begin
        self.pixels16bit := short_Array1d_create(width * height);
        self.inp := pixels;
        self.width := width;
        self.height := height;
        self.inScanline := scanline;
    end;

    procedure Algorithm.scale2x(outp: PIntArray; outScanline: int; turned: boolean);
    var
        pixels: short_Array1d;
        width: int;
        height: int;
        pixCurrLine: int;
        pixPrevLine: int;
        pixNextLine: int;
        outCurrLine: int;
        outNextLine: int;
        outIndex: int;
        pattern: int;
        i: int;
        j: int;
        k: int;
        w: int_Array1d;
        c: int_Array1d;

        procedure pixel00_0(); inline;
        begin
            outp[outCurrLine + outIndex] := c[4];
        end;

        procedure pixel00_10(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[0]);
        end;

        procedure pixel00_11(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel00_12(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[1]);
        end;

        procedure pixel00_20(); inline;
        begin
            outp[outCurrLine + outIndex] := interp2(c[4], c[3], c[1]);
        end;

        procedure pixel00_21(); inline;
        begin
            outp[outCurrLine + outIndex] := interp2(c[4], c[0], c[1]);
        end;

        procedure pixel00_22(); inline;
        begin
            outp[outCurrLine + outIndex] := interp2(c[4], c[0], c[3]);
        end;

        procedure pixel00_60(); inline;
        begin
            outp[outCurrLine + outIndex] := interp6(c[4], c[1], c[3]);
        end;

        procedure pixel00_61(); inline;
        begin
            outp[outCurrLine + outIndex] := interp6(c[4], c[3], c[1]);
        end;

        procedure pixel00_70(); inline;
        begin
            outp[outCurrLine + outIndex] := interp7(c[4], c[3], c[1]);
        end;

        procedure pixel00_90(); inline;
        begin
            outp[outCurrLine + outIndex] := interp9(c[4], c[3], c[1]);
        end;

        procedure pixel00_100(); inline;
        begin
            outp[outCurrLine + outIndex] := interp10(c[4], c[3], c[1]);
        end;

        procedure pixel01_0(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := c[4];
        end;

        procedure pixel01_10(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[2]);
        end;

        procedure pixel01_11(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[1]);
        end;

        procedure pixel01_12(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[5]);
        end;

        procedure pixel01_20(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp2(c[4], c[1], c[5]);
        end;

        procedure pixel01_21(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp2(c[4], c[2], c[5]);
        end;

        procedure pixel01_22(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp2(c[4], c[2], c[1]);
        end;

        procedure pixel01_60(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp6(c[4], c[5], c[1]);
        end;

        procedure pixel01_61(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp6(c[4], c[1], c[5]);
        end;

        procedure pixel01_70(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp7(c[4], c[1], c[5]);
        end;

        procedure pixel01_90(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp9(c[4], c[1], c[5]);
        end;

        procedure pixel01_100(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp10(c[4], c[1], c[5]);
        end;

        procedure pixel10_0(); inline;
        begin
            outp[outNextLine + outIndex] := c[4];
        end;

        procedure pixel10_10(); inline;
        begin
            outp[outNextLine + outIndex] := interp1(c[4], c[6]);
        end;

        procedure pixel10_11(); inline;
        begin
            outp[outNextLine + outIndex] := interp1(c[4], c[7]);
        end;

        procedure pixel10_12(); inline;
        begin
            outp[outNextLine + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel10_20(); inline;
        begin
            outp[outNextLine + outIndex] := interp2(c[4], c[7], c[3]);
        end;

        procedure pixel10_21(); inline;
        begin
            outp[outNextLine + outIndex] := interp2(c[4], c[6], c[3]);
        end;

        procedure pixel10_22(); inline;
        begin
            outp[outNextLine + outIndex] := interp2(c[4], c[6], c[7]);
        end;

        procedure pixel10_60(); inline;
        begin
            outp[outNextLine + outIndex] := interp6(c[4], c[3], c[7]);
        end;

        procedure pixel10_61(); inline;
        begin
            outp[outNextLine + outIndex] := interp6(c[4], c[7], c[3]);
        end;

        procedure pixel10_70(); inline;
        begin
            outp[outNextLine + outIndex] := interp7(c[4], c[7], c[3]);
        end;

        procedure pixel10_90(); inline;
        begin
            outp[outNextLine + outIndex] := interp9(c[4], c[7], c[3]);
        end;

        procedure pixel10_100(); inline;
        begin
            outp[outNextLine + outIndex] := interp10(c[4], c[7], c[3]);
        end;

        procedure pixel11_0(); inline;
        begin
            outp[outNextLine + outIndex + 1] := c[4];
        end;

        procedure pixel11_10(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp1(c[4], c[8]);
        end;

        procedure pixel11_11(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp1(c[4], c[5]);
        end;

        procedure pixel11_12(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp1(c[4], c[7]);
        end;

        procedure pixel11_20(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp2(c[4], c[5], c[7]);
        end;

        procedure pixel11_21(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp2(c[4], c[8], c[7]);
        end;

        procedure pixel11_22(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp2(c[4], c[8], c[5]);
        end;

        procedure pixel11_60(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp6(c[4], c[7], c[5]);
        end;

        procedure pixel11_61(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp6(c[4], c[5], c[7]);
        end;

        procedure pixel11_70(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp7(c[4], c[5], c[7]);
        end;

        procedure pixel11_90(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp9(c[4], c[5], c[7]);
        end;

        procedure pixel11_100(); inline;
        begin
            outp[outNextLine + outIndex + 1] := interp10(c[4], c[5], c[7]);
        end;

    begin
        pixels := self.pixels16bit;
        width := self.width;
        height := self.height;
        transformTo16bit(self.inp, width, height, self.inScanline, pixels);
        w := square3x3w;
        c := square3x3c;
        pixCurrLine := 0;
        outCurrLine := 0;
        if turned then begin
            outCurrLine := outScanline * (height * 2 - 1);
            outScanline := -outScanline;
        end;
        for j := 0 to height - 1 do begin
            if j > 0 then begin
                pixPrevLine := pixCurrLine - width;
            end else begin
                pixPrevLine := pixCurrLine;
            end;
            if j < height - 1 then begin
                pixNextLine := pixCurrLine + width;
            end else begin
                pixNextLine := pixCurrLine;
            end;
            outNextLine := outCurrLine + outScanline;
            outIndex := 0;
            for i := 0 to width - 1 do begin
                w[1] := pixels[pixPrevLine + i] and $ffff;
                w[4] := pixels[pixCurrLine + i] and $ffff;
                w[7] := pixels[pixNextLine + i] and $ffff;
                if i > 0 then begin
                    w[0] := pixels[pixPrevLine + i - 1] and $ffff;
                    w[3] := pixels[pixCurrLine + i - 1] and $ffff;
                    w[6] := pixels[pixNextLine + i - 1] and $ffff;
                end else begin
                    w[0] := w[1];
                    w[3] := w[4];
                    w[6] := w[7];
                end;
                if i < width - 1 then begin
                    w[2] := pixels[pixPrevLine + i + 1] and $ffff;
                    w[5] := pixels[pixCurrLine + i + 1] and $ffff;
                    w[8] := pixels[pixNextLine + i + 1] and $ffff;
                end else begin
                    w[2] := w[1];
                    w[5] := w[4];
                    w[8] := w[7];
                end;
                {
                    p:
                    +---+---+---+
                    | 0 | 1 | 2 |
                    +---+---+---+
                    | 3 | 4 | 5 |
                    +---+---+---+
                    | 6 | 7 | 8 |
                    +---+---+---+
                }
                { Собственно сам алгоритм }
                pattern := 0;
                if diff(w[4], w[0]) <> 0 then begin
                    pattern := pattern or $0001;
                end;
                if diff(w[4], w[1]) <> 0 then begin
                    pattern := pattern or $0002;
                end;
                if diff(w[4], w[2]) <> 0 then begin
                    pattern := pattern or $0004;
                end;
                if diff(w[4], w[3]) <> 0 then begin
                    pattern := pattern or $0008;
                end;
                if diff(w[4], w[5]) <> 0 then begin
                    pattern := pattern or $0010;
                end;
                if diff(w[4], w[6]) <> 0 then begin
                    pattern := pattern or $0020;
                end;
                if diff(w[4], w[7]) <> 0 then begin
                    pattern := pattern or $0040;
                end;
                if diff(w[4], w[8]) <> 0 then begin
                    pattern := pattern or $0080;
                end;
                for k := 0 to 8 do begin
                    c[k] := transformTo32bit(w[k]);
                end;
                case pattern of
                0, 1, 4, 5, 32, 33, 36, 37, 128, 129, 132, 133, 160, 161, 164, 165: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_20();
                    pixel11_20();
                end;
                2, 34, 130, 162: begin
                    pixel00_22();
                    pixel01_21();
                    pixel10_20();
                    pixel11_20();
                end;
                3, 35, 131, 163: begin
                    pixel00_11();
                    pixel01_21();
                    pixel10_20();
                    pixel11_20();
                end;
                6, 38, 134, 166: begin
                    pixel00_22();
                    pixel01_12();
                    pixel10_20();
                    pixel11_20();
                end;
                7, 39, 135: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_20();
                    pixel11_20();
                end;
                8, 12, 136, 140: begin
                    pixel00_21();
                    pixel01_20();
                    pixel10_22();
                    pixel11_20();
                end;
                9, 13, 137, 141: begin
                    pixel00_12();
                    pixel01_20();
                    pixel10_22();
                    pixel11_20();
                end;
                10, 138: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    pixel10_22();
                    pixel11_20();
                end;
                11, 139: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    pixel10_22();
                    pixel11_20();
                end;
                14, 142: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                        pixel01_12();
                    end else begin
                        pixel00_90();
                        pixel01_61();
                    end;
                    pixel10_22();
                    pixel11_20();
                end;
                15, 143: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_12();
                    end else begin
                        pixel00_90();
                        pixel01_61();
                    end;
                    pixel10_22();
                    pixel11_20();
                end;
                16, 17, 48, 49: begin
                    pixel00_20();
                    pixel01_22();
                    pixel10_20();
                    pixel11_21();
                end;
                18, 50: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_20();
                    pixel11_21();
                end;
                19, 51: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_11();
                        pixel01_10();
                    end else begin
                        pixel00_60();
                        pixel01_90();
                    end;
                    pixel10_20();
                    pixel11_21();
                end;
                20, 21, 52, 53: begin
                    pixel00_20();
                    pixel01_11();
                    pixel10_20();
                    pixel11_21();
                end;
                22, 54: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_20();
                    pixel11_21();
                end;
                23, 55: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_11();
                        pixel01_0();
                    end else begin
                        pixel00_60();
                        pixel01_90();
                    end;
                    pixel10_20();
                    pixel11_21();
                end;
                24: begin
                    pixel00_21();
                    pixel01_22();
                    pixel10_22();
                    pixel11_21();
                end;
                25: begin
                    pixel00_12();
                    pixel01_22();
                    pixel10_22();
                    pixel11_21();
                end;
                26, 31: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_22();
                    pixel11_21();
                end;
                27: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_10();
                    pixel10_22();
                    pixel11_21();
                end;
                28: begin
                    pixel00_21();
                    pixel01_11();
                    pixel10_22();
                    pixel11_21();
                end;
                29: begin
                    pixel00_12();
                    pixel01_11();
                    pixel10_22();
                    pixel11_21();
                end;
                30: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_22();
                    pixel11_21();
                end;
                40, 44, 168, 172: begin
                    pixel00_21();
                    pixel01_20();
                    pixel10_11();
                    pixel11_20();
                end;
                41, 45, 169: begin
                    pixel00_12();
                    pixel01_20();
                    pixel10_11();
                    pixel11_20();
                end;
                42, 170: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                        pixel10_11();
                    end else begin
                        pixel00_90();
                        pixel10_60();
                    end;
                    pixel01_21();
                    pixel11_20();
                end;
                43, 171: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel10_11();
                    end else begin
                        pixel00_90();
                        pixel10_60();
                    end;
                    pixel01_21();
                    pixel11_20();
                end;
                46, 174: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    pixel01_12();
                    pixel10_11();
                    pixel11_20();
                end;
                47, 175: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    pixel01_12();
                    pixel10_11();
                    pixel11_20();
                end;
                56: begin
                    pixel00_21();
                    pixel01_22();
                    pixel10_11();
                    pixel11_21();
                end;
                57: begin
                    pixel00_12();
                    pixel01_22();
                    pixel10_11();
                    pixel11_21();
                end;
                58: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_11();
                    pixel11_21();
                end;
                59: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_11();
                    pixel11_21();
                end;
                60: begin
                    pixel00_21();
                    pixel01_11();
                    pixel10_11();
                    pixel11_21();
                end;
                61: begin
                    pixel00_12();
                    pixel01_11();
                    pixel10_11();
                    pixel11_21();
                end;
                62: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_11();
                    pixel11_21();
                end;
                63: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_11();
                    pixel11_21();
                end;
                64, 65, 68, 69: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_21();
                    pixel11_22();
                end;
                66: begin
                    pixel00_22();
                    pixel01_21();
                    pixel10_21();
                    pixel11_22();
                end;
                67: begin
                    pixel00_11();
                    pixel01_21();
                    pixel10_21();
                    pixel11_22();
                end;
                70: begin
                    pixel00_22();
                    pixel01_12();
                    pixel10_21();
                    pixel11_22();
                end;
                71: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_21();
                    pixel11_22();
                end;
                72, 76: begin
                    pixel00_21();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                73, 77: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_12();
                        pixel10_10();
                    end else begin
                        pixel00_61();
                        pixel10_90();
                    end;
                    pixel01_20();
                    pixel11_22();
                end;
                74, 107: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                75: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    pixel10_10();
                    pixel11_22();
                end;
                78: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    pixel11_22();
                end;
                79: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    pixel11_22();
                end;
                80, 81: begin
                    pixel00_20();
                    pixel01_22();
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_20();
                    end;
                end;
                82, 214: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                83: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                84, 85: begin
                    pixel00_20();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel01_11();
                        pixel11_10();
                    end else begin
                        pixel01_60();
                        pixel11_90();
                    end;
                    pixel10_21();
                end;
                86: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_21();
                    pixel11_10();
                end;
                87: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                88, 248: begin
                    pixel00_21();
                    pixel01_22();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                89: begin
                    pixel00_12();
                    pixel01_22();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                90: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                91: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                92: begin
                    pixel00_21();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                93: begin
                    pixel00_12();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                94: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                95: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_10();
                    pixel11_10();
                end;
                96, 97, 100, 101: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_12();
                    pixel11_22();
                end;
                98: begin
                    pixel00_22();
                    pixel01_21();
                    pixel10_12();
                    pixel11_22();
                end;
                99: begin
                    pixel00_11();
                    pixel01_21();
                    pixel10_12();
                    pixel11_22();
                end;
                102: begin
                    pixel00_22();
                    pixel01_12();
                    pixel10_12();
                    pixel11_22();
                end;
                103: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_12();
                    pixel11_22();
                end;
                104, 108: begin
                    pixel00_21();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                105, 109: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_12();
                        pixel10_0();
                    end else begin
                        pixel00_61();
                        pixel10_90();
                    end;
                    pixel01_20();
                    pixel11_22();
                end;
                106: begin
                    pixel00_10();
                    pixel01_21();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                110: begin
                    pixel00_10();
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                111: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_22();
                end;
                112, 113: begin
                    pixel00_20();
                    pixel01_22();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel10_12();
                        pixel11_10();
                    end else begin
                        pixel10_61();
                        pixel11_90();
                    end;
                end;
                114: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                115: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                116, 117: begin
                    pixel00_20();
                    pixel01_11();
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                118: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_12();
                    pixel11_10();
                end;
                119: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_11();
                        pixel01_0();
                    end else begin
                        pixel00_60();
                        pixel01_90();
                    end;
                    pixel10_12();
                    pixel11_10();
                end;
                120: begin
                    pixel00_21();
                    pixel01_22();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_10();
                end;
                121: begin
                    pixel00_12();
                    pixel01_22();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                122: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_10();
                    end else begin
                        pixel11_70();
                    end;
                end;
                123: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_10();
                end;
                124: begin
                    pixel00_21();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_10();
                end;
                125: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_12();
                        pixel10_0();
                    end else begin
                        pixel00_61();
                        pixel10_90();
                    end;
                    pixel01_11();
                    pixel11_10();
                end;
                126: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_10();
                end;
                127: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_10();
                end;
                144, 145, 176, 177: begin
                    pixel00_20();
                    pixel01_22();
                    pixel10_20();
                    pixel11_12();
                end;
                146, 178: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                        pixel11_12();
                    end else begin
                        pixel01_90();
                        pixel11_61();
                    end;
                    pixel10_20();
                end;
                147, 179: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_20();
                    pixel11_12();
                end;
                148, 149, 180: begin
                    pixel00_20();
                    pixel01_11();
                    pixel10_20();
                    pixel11_12();
                end;
                150, 182: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                        pixel11_12();
                    end else begin
                        pixel01_90();
                        pixel11_61();
                    end;
                    pixel10_20();
                end;
                151, 183: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_20();
                    pixel11_12();
                end;
                152: begin
                    pixel00_21();
                    pixel01_22();
                    pixel10_22();
                    pixel11_12();
                end;
                153: begin
                    pixel00_12();
                    pixel01_22();
                    pixel10_22();
                    pixel11_12();
                end;
                154: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_22();
                    pixel11_12();
                end;
                155: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_10();
                    pixel10_22();
                    pixel11_12();
                end;
                156: begin
                    pixel00_21();
                    pixel01_11();
                    pixel10_22();
                    pixel11_12();
                end;
                157: begin
                    pixel00_12();
                    pixel01_11();
                    pixel10_22();
                    pixel11_12();
                end;
                158: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_22();
                    pixel11_12();
                end;
                159: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_22();
                    pixel11_12();
                end;
                167: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_20();
                    pixel11_20();
                end;
                173: begin
                    pixel00_12();
                    pixel01_20();
                    pixel10_11();
                    pixel11_20();
                end;
                181: begin
                    pixel00_20();
                    pixel01_11();
                    pixel10_20();
                    pixel11_12();
                end;
                184: begin
                    pixel00_21();
                    pixel01_22();
                    pixel10_11();
                    pixel11_12();
                end;
                185: begin
                    pixel00_12();
                    pixel01_22();
                    pixel10_11();
                    pixel11_12();
                end;
                186: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_11();
                    pixel11_12();
                end;
                187: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel10_11();
                    end else begin
                        pixel00_90();
                        pixel10_60();
                    end;
                    pixel01_10();
                    pixel11_12();
                end;
                188: begin
                    pixel00_21();
                    pixel01_11();
                    pixel10_11();
                    pixel11_12();
                end;
                189: begin
                    pixel00_12();
                    pixel01_11();
                    pixel10_11();
                    pixel11_12();
                end;
                190: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                        pixel11_12();
                    end else begin
                        pixel01_90();
                        pixel11_61();
                    end;
                    pixel10_11();
                end;
                191: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_11();
                    pixel11_12();
                end;
                192, 193, 196, 197: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_21();
                    pixel11_11();
                end;
                194: begin
                    pixel00_22();
                    pixel01_21();
                    pixel10_21();
                    pixel11_11();
                end;
                195: begin
                    pixel00_11();
                    pixel01_21();
                    pixel10_21();
                    pixel11_11();
                end;
                198: begin
                    pixel00_22();
                    pixel01_12();
                    pixel10_21();
                    pixel11_11();
                end;
                199: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_21();
                    pixel11_11();
                end;
                200, 204: begin
                    pixel00_21();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                        pixel11_11();
                    end else begin
                        pixel10_90();
                        pixel11_60();
                    end;
                end;
                201, 205: begin
                    pixel00_12();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    pixel11_11();
                end;
                202: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    pixel01_21();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    pixel11_11();
                end;
                203: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    pixel10_10();
                    pixel11_11();
                end;
                206: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    pixel11_11();
                end;
                207: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_12();
                    end else begin
                        pixel00_90();
                        pixel01_61();
                    end;
                    pixel10_10();
                    pixel11_11();
                end;
                208, 209: begin
                    pixel00_20();
                    pixel01_22();
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                210: begin
                    pixel00_22();
                    pixel01_10();
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                211: begin
                    pixel00_11();
                    pixel01_10();
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                212, 213: begin
                    pixel00_20();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel01_11();
                        pixel11_0();
                    end else begin
                        pixel01_60();
                        pixel11_90();
                    end;
                    pixel10_21();
                end;
                215: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_21();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                216: begin
                    pixel00_21();
                    pixel01_22();
                    pixel10_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                217: begin
                    pixel00_12();
                    pixel01_22();
                    pixel10_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                218: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                219: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_10();
                    pixel10_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                220: begin
                    pixel00_21();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_10();
                    end else begin
                        pixel10_70();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                221: begin
                    pixel00_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel01_11();
                        pixel11_0();
                    end else begin
                        pixel01_60();
                        pixel11_90();
                    end;
                    pixel10_10();
                end;
                222: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                223: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                224, 225, 228: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_12();
                    pixel11_11();
                end;
                226: begin
                    pixel00_22();
                    pixel01_21();
                    pixel10_12();
                    pixel11_11();
                end;
                227: begin
                    pixel00_11();
                    pixel01_21();
                    pixel10_12();
                    pixel11_11();
                end;
                229: begin
                    pixel00_20();
                    pixel01_20();
                    pixel10_12();
                    pixel11_11();
                end;
                230: begin
                    pixel00_22();
                    pixel01_12();
                    pixel10_12();
                    pixel11_11();
                end;
                231: begin
                    pixel00_11();
                    pixel01_12();
                    pixel10_12();
                    pixel11_11();
                end;
                232, 236: begin
                    pixel00_21();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                        pixel11_11();
                    end else begin
                        pixel10_90();
                        pixel11_60();
                    end;
                end;
                233, 237: begin
                    pixel00_12();
                    pixel01_20();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    pixel11_11();
                end;
                234: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_10();
                    end else begin
                        pixel00_70();
                    end;
                    pixel01_21();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    pixel11_11();
                end;
                235: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_21();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    pixel11_11();
                end;
                238: begin
                    pixel00_10();
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                        pixel11_11();
                    end else begin
                        pixel10_90();
                        pixel11_60();
                    end;
                end;
                239: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    pixel01_12();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    pixel11_11();
                end;
                240, 241: begin
                    pixel00_20();
                    pixel01_22();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel10_12();
                        pixel11_0();
                    end else begin
                        pixel10_61();
                        pixel11_90();
                    end;
                end;
                242: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_10();
                    end else begin
                        pixel01_70();
                    end;
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                243: begin
                    pixel00_11();
                    pixel01_10();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel10_12();
                        pixel11_0();
                    end else begin
                        pixel10_61();
                        pixel11_90();
                    end;
                end;
                244, 245: begin
                    pixel00_20();
                    pixel01_11();
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                246: begin
                    pixel00_22();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                247: begin
                    pixel00_11();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    pixel10_12();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                249: begin
                    pixel00_12();
                    pixel01_22();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                250: begin
                    pixel00_10();
                    pixel01_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                251: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_20();
                    end;
                end;
                252: begin
                    pixel00_21();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                253: begin
                    pixel00_12();
                    pixel01_11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                254: begin
                    pixel00_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_20();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_20();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                255: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_100();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_0();
                    end else begin
                        pixel01_100();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_0();
                    end else begin
                        pixel10_100();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel11_0();
                    end else begin
                        pixel11_100();
                    end;
                end;
                end;
                { Конец алгоритма }
                inc(outIndex, 2);
            end;
            inc(pixCurrLine, width);
            inc(outCurrLine, outScanline * 2);
        end;
        returnToFPU();
    end;

    procedure Algorithm.scale3x(outp: PIntArray; outScanline: int; turned: boolean);
    var
        pixels: short_Array1d;
        width: int;
        height: int;
        pixCurrLine: int;
        pixPrevLine: int;
        pixNextLine: int;
        outCurrLine: int;
        outNext1Line: int;
        outNext2Line: int;
        outIndex: int;
        pattern: int;
        i: int;
        j: int;
        k: int;
        w: int_Array1d;
        c: int_Array1d;

        procedure pixel00_1m(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[0]);
        end;

        procedure pixel00_1u(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[1]);
        end;

        procedure pixel00_1l(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel00_2(); inline;
        begin
            outp[outCurrLine + outIndex] := interp2(c[4], c[3], c[1]);
        end;

        procedure pixel00_4(); inline;
        begin
            outp[outCurrLine + outIndex] := interp4(c[4], c[3], c[1]);
        end;

        procedure pixel00_5(); inline;
        begin
            outp[outCurrLine + outIndex] := interp5(c[3], c[1]);
        end;

        procedure pixel00_c(); inline;
        begin
            outp[outCurrLine + outIndex] := c[4];
        end;

        procedure pixel01_1(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[1]);
        end;

        procedure pixel01_3(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp3(c[4], c[1]);
        end;

        procedure pixel01_6(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[1], c[4]);
        end;

        procedure pixel01_c(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := c[4];
        end;

        procedure pixel02_1m(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[4], c[2]);
        end;

        procedure pixel02_1u(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[4], c[1]);
        end;

        procedure pixel02_1r(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[4], c[5]);
        end;

        procedure pixel02_2(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp2(c[4], c[1], c[5]);
        end;

        procedure pixel02_4(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp4(c[4], c[1], c[5]);
        end;

        procedure pixel02_5(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp5(c[1], c[5]);
        end;

        procedure pixel02_c(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := c[4];
        end;

        procedure pixel10_1(); inline;
        begin
            outp[outNext1Line + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel10_3(); inline;
        begin
            outp[outNext1Line + outIndex] := interp3(c[4], c[3]);
        end;

        procedure pixel10_6(); inline;
        begin
            outp[outNext1Line + outIndex] := interp1(c[3], c[4]);
        end;

        procedure pixel10_c(); inline;
        begin
            outp[outNext1Line + outIndex] := c[4];
        end;

        procedure pixel11(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := c[4];
        end;

        procedure pixel12_1(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp1(c[4], c[5]);
        end;

        procedure pixel12_3(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp3(c[4], c[5]);
        end;

        procedure pixel12_6(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp1(c[5], c[4]);
        end;

        procedure pixel12_c(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := c[4];
        end;

        procedure pixel20_1m(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[4], c[6]);
        end;

        procedure pixel20_1d(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[4], c[7]);
        end;

        procedure pixel20_1l(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel20_2(); inline;
        begin
            outp[outNext2Line + outIndex] := interp2(c[4], c[7], c[3]);
        end;

        procedure pixel20_4(); inline;
        begin
            outp[outNext2Line + outIndex] := interp4(c[4], c[7], c[3]);
        end;

        procedure pixel20_5(); inline;
        begin
            outp[outNext2Line + outIndex] := interp5(c[7], c[3]);
        end;

        procedure pixel20_c(); inline;
        begin
            outp[outNext2Line + outIndex] := c[4];
        end;

        procedure pixel21_1(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp1(c[4], c[7]);
        end;

        procedure pixel21_3(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp3(c[4], c[7]);
        end;

        procedure pixel21_6(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp1(c[7], c[4]);
        end;

        procedure pixel21_c(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := c[4];
        end;

        procedure pixel22_1m(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp1(c[4], c[8]);
        end;

        procedure pixel22_1d(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp1(c[4], c[7]);
        end;

        procedure pixel22_1r(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp1(c[4], c[5]);
        end;

        procedure pixel22_2(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp2(c[4], c[5], c[7]);
        end;

        procedure pixel22_4(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp4(c[4], c[5], c[7]);
        end;

        procedure pixel22_5(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp5(c[5], c[7]);
        end;

        procedure pixel22_c(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := c[4];
        end;

    begin
        pixels := self.pixels16bit;
        width := self.width;
        height := self.height;
        transformTo16bit(self.inp, width, height, self.inScanline, pixels);
        w := square3x3w;
        c := square3x3c;
        pixCurrLine := 0;
        outCurrLine := 0;
        if turned then begin
            outCurrLine := outScanline * (height * 3 - 1);
            outScanline := -outScanline;
        end;
        for j := 0 to height - 1 do begin
            if j > 0 then begin
                pixPrevLine := pixCurrLine - width;
            end else begin
                pixPrevLine := pixCurrLine;
            end;
            if j < height - 1 then begin
                pixNextLine := pixCurrLine + width;
            end else begin
                pixNextLine := pixCurrLine;
            end;
            outNext1Line := outCurrLine + outScanline;
            outNext2Line := outNext1Line + outScanline;
            outIndex := 0;
            for i := 0 to width - 1 do begin
                w[1] := pixels[pixPrevLine + i] and $ffff;
                w[4] := pixels[pixCurrLine + i] and $ffff;
                w[7] := pixels[pixNextLine + i] and $ffff;
                if i > 0 then begin
                    w[0] := pixels[pixPrevLine + i - 1] and $ffff;
                    w[3] := pixels[pixCurrLine + i - 1] and $ffff;
                    w[6] := pixels[pixNextLine + i - 1] and $ffff;
                end else begin
                    w[0] := w[1];
                    w[3] := w[4];
                    w[6] := w[7];
                end;
                if i < width - 1 then begin
                    w[2] := pixels[pixPrevLine + i + 1] and $ffff;
                    w[5] := pixels[pixCurrLine + i + 1] and $ffff;
                    w[8] := pixels[pixNextLine + i + 1] and $ffff;
                end else begin
                    w[2] := w[1];
                    w[5] := w[4];
                    w[8] := w[7];
                end;
                {
                    p:
                    +---+---+---+
                    | 0 | 1 | 2 |
                    +---+---+---+
                    | 3 | 4 | 5 |
                    +---+---+---+
                    | 6 | 7 | 8 |
                    +---+---+---+
                }
                { Собственно сам алгоритм }
                pattern := 0;
                if diff(w[4], w[0]) <> 0 then begin
                    pattern := pattern or $0001;
                end;
                if diff(w[4], w[1]) <> 0 then begin
                    pattern := pattern or $0002;
                end;
                if diff(w[4], w[2]) <> 0 then begin
                    pattern := pattern or $0004;
                end;
                if diff(w[4], w[3]) <> 0 then begin
                    pattern := pattern or $0008;
                end;
                if diff(w[4], w[5]) <> 0 then begin
                    pattern := pattern or $0010;
                end;
                if diff(w[4], w[6]) <> 0 then begin
                    pattern := pattern or $0020;
                end;
                if diff(w[4], w[7]) <> 0 then begin
                    pattern := pattern or $0040;
                end;
                if diff(w[4], w[8]) <> 0 then begin
                    pattern := pattern or $0080;
                end;
                for k := 0 to 8 do begin
                    c[k] := transformTo32bit(w[k]);
                end;
                case pattern of
                0, 1, 4, 5, 32, 33, 36, 37, 128, 129, 132, 133, 160, 161, 164, 165: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                2, 34, 130, 162: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                3, 35, 131, 163: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                6, 38, 134, 166: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                7, 39, 135: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                8, 12, 136, 140: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                9, 13, 137, 141: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                10, 138: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                11, 139: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                14, 142: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                        pixel01_c();
                        pixel02_1r();
                        pixel10_c();
                    end else begin
                        pixel00_5();
                        pixel01_6();
                        pixel02_2();
                        pixel10_1();
                    end;
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                15, 143: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel02_1r();
                        pixel10_c();
                    end else begin
                        pixel00_5();
                        pixel01_6();
                        pixel02_2();
                        pixel10_1();
                    end;
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_2();
                end;
                16, 17, 48, 49: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                18, 50: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_1m();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                19, 51: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_1l();
                        pixel01_c();
                        pixel02_1m();
                        pixel12_c();
                    end else begin
                        pixel00_2();
                        pixel01_6();
                        pixel02_5();
                        pixel12_1();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                20, 21, 52, 53: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                22, 54: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                23, 55: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_1l();
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel00_2();
                        pixel01_6();
                        pixel02_5();
                        pixel12_1();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1m();
                end;
                24: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                25: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                26, 31: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel10_3();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel11();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                27: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                28: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                29: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                30: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1m();
                end;
                40, 44, 168, 172: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_2();
                end;
                41, 45, 169: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_2();
                end;
                42, 170: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                        pixel01_c();
                        pixel10_c();
                        pixel20_1d();
                    end else begin
                        pixel00_5();
                        pixel01_1();
                        pixel10_6();
                        pixel20_2();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel21_1();
                    pixel22_2();
                end;
                43, 171: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                        pixel20_1d();
                    end else begin
                        pixel00_5();
                        pixel01_1();
                        pixel10_6();
                        pixel20_2();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel21_1();
                    pixel22_2();
                end;
                46, 174: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_2();
                end;
                47, 175: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_2();
                end;
                56: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                57: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                58: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                59: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                60: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                61: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                62: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                63: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1m();
                end;
                64, 65, 68, 69: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                66: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                67: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                70: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                71: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                72, 76: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_1m();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                73, 77: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_1u();
                        pixel10_c();
                        pixel20_1m();
                        pixel21_c();
                    end else begin
                        pixel00_2();
                        pixel10_6();
                        pixel20_5();
                        pixel21_1();
                    end;
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                    pixel22_1m();
                end;
                74, 107: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                    end;
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                75: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                78: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1m();
                end;
                79: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1r();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1m();
                end;
                80, 81: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_1m();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                82, 214: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                83: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                84, 85: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel02_1u();
                        pixel12_c();
                        pixel21_c();
                        pixel22_1m();
                    end else begin
                        pixel02_2();
                        pixel12_6();
                        pixel21_1();
                        pixel22_5();
                    end;
                    pixel00_2();
                    pixel01_1();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                end;
                86: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                87: begin
                    pixel00_1l();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                88, 248: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel22_4();
                    end;
                end;
                89: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                90: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                91: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                92: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                93: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                94: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_c();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                95: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel10_3();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel11();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1m();
                end;
                96, 97, 100, 101: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                98: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                99: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                102: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                103: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                104, 108: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                105, 109: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_1u();
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel00_2();
                        pixel10_6();
                        pixel20_5();
                        pixel21_1();
                    end;
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                    pixel22_1m();
                end;
                106: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                110: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                111: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                112, 113: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel20_1l();
                        pixel21_c();
                        pixel22_1m();
                    end else begin
                        pixel12_1();
                        pixel20_2();
                        pixel21_6();
                        pixel22_5();
                    end;
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                end;
                114: begin
                    pixel00_1m();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                115: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                116, 117: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                118: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                119: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_1l();
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel00_2();
                        pixel01_6();
                        pixel02_5();
                        pixel12_1();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1m();
                end;
                120: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                121: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                122: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_1m();
                    end else begin
                        pixel22_2();
                    end;
                end;
                123: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                    end;
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                124: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                125: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_1u();
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel00_2();
                        pixel10_6();
                        pixel20_5();
                        pixel21_1();
                    end;
                    pixel01_1();
                    pixel02_1u();
                    pixel11();
                    pixel12_c();
                    pixel22_1m();
                end;
                126: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                127: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_2();
                        pixel01_3();
                        pixel10_3();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1m();
                end;
                144, 145, 176, 177: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1d();
                end;
                146, 178: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_1m();
                        pixel12_c();
                        pixel22_1d();
                    end else begin
                        pixel01_1();
                        pixel02_5();
                        pixel12_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                end;
                147, 179: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1d();
                end;
                148, 149, 180: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1d();
                end;
                150, 182: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                        pixel22_1d();
                    end else begin
                        pixel01_1();
                        pixel02_5();
                        pixel12_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_2();
                    pixel21_1();
                end;
                151, 183: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1d();
                end;
                152: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                153: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                154: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                155: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                156: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                157: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                158: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                        pixel12_3();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                159: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel10_3();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    pixel21_1();
                    pixel22_1d();
                end;
                167: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_2();
                    pixel21_1();
                    pixel22_2();
                end;
                173: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_2();
                end;
                181: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_2();
                    pixel21_1();
                    pixel22_1d();
                end;
                184: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                185: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                186: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                187: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                        pixel20_1d();
                    end else begin
                        pixel00_5();
                        pixel01_1();
                        pixel10_6();
                        pixel20_2();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_c();
                    pixel21_1();
                    pixel22_1d();
                end;
                188: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                189: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                190: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                        pixel22_1d();
                    end else begin
                        pixel01_1();
                        pixel02_5();
                        pixel12_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel10_c();
                    pixel11();
                    pixel20_1d();
                    pixel21_1();
                end;
                191: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1d();
                    pixel21_1();
                    pixel22_1d();
                end;
                192, 193, 196, 197: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                194: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                195: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                198: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                199: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                200, 204: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_1m();
                        pixel21_c();
                        pixel22_1r();
                    end else begin
                        pixel10_1();
                        pixel20_5();
                        pixel21_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                end;
                201, 205: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                202: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                203: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                206: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                207: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel02_1r();
                        pixel10_c();
                    end else begin
                        pixel00_5();
                        pixel01_6();
                        pixel02_2();
                        pixel10_1();
                    end;
                    pixel11();
                    pixel12_1();
                    pixel20_1m();
                    pixel21_c();
                    pixel22_1r();
                end;
                208, 209: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                210: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                211: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                212, 213: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel02_1u();
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel02_2();
                        pixel12_6();
                        pixel21_1();
                        pixel22_5();
                    end;
                    pixel00_2();
                    pixel01_1();
                    pixel10_1();
                    pixel11();
                    pixel20_1m();
                end;
                215: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                216: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                217: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                218: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                219: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                        pixel10_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                220: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_1m();
                    end else begin
                        pixel20_2();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                221: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel02_1u();
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel02_2();
                        pixel12_6();
                        pixel21_1();
                        pixel22_5();
                    end;
                    pixel00_1u();
                    pixel01_1();
                    pixel10_c();
                    pixel11();
                    pixel20_1m();
                end;
                222: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                223: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel10_c();
                    end else begin
                        pixel00_4();
                        pixel10_3();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                        pixel12_c();
                    end else begin
                        pixel01_3();
                        pixel02_2();
                        pixel12_3();
                    end;
                    pixel11();
                    pixel20_1m();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                224, 225, 228: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                226: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                227: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                229: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_2();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                230: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                231: begin
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1r();
                    pixel10_1();
                    pixel11();
                    pixel12_1();
                    pixel20_1l();
                    pixel21_c();
                    pixel22_1r();
                end;
                232, 236: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                        pixel22_1r();
                    end else begin
                        pixel10_1();
                        pixel20_5();
                        pixel21_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel01_1();
                    pixel02_2();
                    pixel11();
                    pixel12_1();
                end;
                233, 237: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_2();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                234: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_1m();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1m();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                        pixel21_3();
                    end;
                    pixel22_1r();
                end;
                235: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                    end;
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                238: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                        pixel22_1r();
                    end else begin
                        pixel10_1();
                        pixel20_5();
                        pixel21_6();
                        pixel22_2();
                    end;
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1r();
                    pixel11();
                    pixel12_1();
                end;
                239: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    pixel02_1r();
                    pixel10_c();
                    pixel11();
                    pixel12_1();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    pixel22_1r();
                end;
                240, 241: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel20_1l();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_1();
                        pixel20_2();
                        pixel21_6();
                        pixel22_5();
                    end;
                    pixel00_2();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                end;
                242: begin
                    pixel00_1m();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_1m();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel20_1l();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_4();
                    end;
                end;
                243: begin
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel20_1l();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_1();
                        pixel20_2();
                        pixel21_6();
                        pixel22_5();
                    end;
                    pixel00_1l();
                    pixel01_c();
                    pixel02_1m();
                    pixel10_1();
                    pixel11();
                end;
                244, 245: begin
                    pixel00_2();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                246: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                247: begin
                    pixel00_1l();
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_1();
                    pixel11();
                    pixel12_c();
                    pixel20_1l();
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                249: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1m();
                    pixel10_c();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel22_4();
                    end;
                end;
                250: begin
                    pixel00_1m();
                    pixel01_c();
                    pixel02_1m();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel22_4();
                    end;
                end;
                251: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                        pixel01_c();
                    end else begin
                        pixel00_4();
                        pixel01_3();
                    end;
                    pixel02_1m();
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                        pixel21_c();
                    end else begin
                        pixel10_3();
                        pixel20_2();
                        pixel21_3();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel22_4();
                    end;
                end;
                252: begin
                    pixel00_1m();
                    pixel01_1();
                    pixel02_1u();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                253: begin
                    pixel00_1u();
                    pixel01_1();
                    pixel02_1u();
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                254: begin
                    pixel00_1m();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel01_c();
                        pixel02_c();
                    end else begin
                        pixel01_3();
                        pixel02_4();
                    end;
                    pixel11();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel10_c();
                        pixel20_c();
                    end else begin
                        pixel10_3();
                        pixel20_4();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel12_c();
                        pixel21_c();
                        pixel22_c();
                    end else begin
                        pixel12_3();
                        pixel21_3();
                        pixel22_2();
                    end;
                end;
                255: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_c();
                    end else begin
                        pixel00_2();
                    end;
                    pixel01_c();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_c();
                    end else begin
                        pixel02_2();
                    end;
                    pixel10_c();
                    pixel11();
                    pixel12_c();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_c();
                    end else begin
                        pixel20_2();
                    end;
                    pixel21_c();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_c();
                    end else begin
                        pixel22_2();
                    end;
                end;
                end;
                { Конец алгоритма }
                inc(outIndex, 3);
            end;
            inc(pixCurrLine, width);
            inc(outCurrLine, outScanline * 3);
        end;
        returnToFPU();
    end;

    procedure Algorithm.scale4x(outp: PIntArray; outScanline: int; turned: boolean);
    var
        pixels: short_Array1d;
        width: int;
        height: int;
        pixCurrLine: int;
        pixPrevLine: int;
        pixNextLine: int;
        outCurrLine: int;
        outNext1Line: int;
        outNext2Line: int;
        outNext3Line: int;
        outIndex: int;
        pattern: int;
        i: int;
        j: int;
        k: int;
        w: int_Array1d;
        c: int_Array1d;

        procedure pixel00_0(); inline;
        begin
            outp[outCurrLine + outIndex] := c[4];
        end;

        procedure pixel00_11(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel00_12(); inline;
        begin
            outp[outCurrLine + outIndex] := interp1(c[4], c[1]);
        end;

        procedure pixel00_20(); inline;
        begin
            outp[outCurrLine + outIndex] := interp2(c[4], c[1], c[3]);
        end;

        procedure pixel00_50(); inline;
        begin
            outp[outCurrLine + outIndex] := interp5(c[1], c[3]);
        end;

        procedure pixel00_80(); inline;
        begin
            outp[outCurrLine + outIndex] := interp8(c[4], c[0]);
        end;

        procedure pixel00_81(); inline;
        begin
            outp[outCurrLine + outIndex] := interp8(c[4], c[3]);
        end;

        procedure pixel00_82(); inline;
        begin
            outp[outCurrLine + outIndex] := interp8(c[4], c[1]);
        end;

        procedure pixel01_0(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := c[4];
        end;

        procedure pixel01_10(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[0]);
        end;

        procedure pixel01_12(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[4], c[1]);
        end;

        procedure pixel01_14(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp1(c[1], c[4]);
        end;

        procedure pixel01_21(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp2(c[1], c[4], c[3]);
        end;

        procedure pixel01_31(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp3(c[4], c[3]);
        end;

        procedure pixel01_50(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp5(c[1], c[4]);
        end;

        procedure pixel01_60(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp6(c[4], c[1], c[3]);
        end;

        procedure pixel01_61(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp6(c[4], c[1], c[0]);
        end;

        procedure pixel01_82(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp8(c[4], c[1]);
        end;

        procedure pixel01_83(); inline;
        begin
            outp[outCurrLine + outIndex + 1] := interp8(c[1], c[3]);
        end;

        procedure pixel02_0(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := c[4];
        end;

        procedure pixel02_10(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[4], c[2]);
        end;

        procedure pixel02_11(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[4], c[1]);
        end;

        procedure pixel02_13(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp1(c[1], c[4]);
        end;

        procedure pixel02_21(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp2(c[1], c[4], c[5]);
        end;

        procedure pixel02_32(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp3(c[4], c[5]);
        end;

        procedure pixel02_50(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp5(c[1], c[4]);
        end;

        procedure pixel02_60(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp6(c[4], c[1], c[5]);
        end;

        procedure pixel02_61(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp6(c[4], c[1], c[2]);
        end;

        procedure pixel02_81(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp8(c[4], c[1]);
        end;

        procedure pixel02_83(); inline;
        begin
            outp[outCurrLine + outIndex + 2] := interp8(c[1], c[5]);
        end;

        procedure pixel03_0(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := c[4];
        end;

        procedure pixel03_11(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp1(c[4], c[1]);
        end;

        procedure pixel03_12(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp1(c[4], c[5]);
        end;

        procedure pixel03_20(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp2(c[4], c[1], c[5]);
        end;

        procedure pixel03_50(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp5(c[1], c[5]);
        end;

        procedure pixel03_80(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp8(c[4], c[2]);
        end;

        procedure pixel03_81(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp8(c[4], c[1]);
        end;

        procedure pixel03_82(); inline;
        begin
            outp[outCurrLine + outIndex + 3] := interp8(c[4], c[5]);
        end;

        procedure pixel10_0(); inline;
        begin
            outp[outNext1Line + outIndex] := c[4];
        end;

        procedure pixel10_10(); inline;
        begin
            outp[outNext1Line + outIndex] := interp1(c[4], c[0]);
        end;

        procedure pixel10_11(); inline;
        begin
            outp[outNext1Line + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel10_13(); inline;
        begin
            outp[outNext1Line + outIndex] := interp1(c[3], c[4]);
        end;

        procedure pixel10_21(); inline;
        begin
            outp[outNext1Line + outIndex] := interp2(c[3], c[4], c[1]);
        end;

        procedure pixel10_32(); inline;
        begin
            outp[outNext1Line + outIndex] := interp3(c[4], c[1]);
        end;

        procedure pixel10_50(); inline;
        begin
            outp[outNext1Line + outIndex] := interp5(c[3], c[4]);
        end;

        procedure pixel10_60(); inline;
        begin
            outp[outNext1Line + outIndex] := interp6(c[4], c[3], c[1]);
        end;

        procedure pixel10_61(); inline;
        begin
            outp[outNext1Line + outIndex] := interp6(c[4], c[3], c[0]);
        end;

        procedure pixel10_81(); inline;
        begin
            outp[outNext1Line + outIndex] := interp8(c[4], c[3]);
        end;

        procedure pixel10_83(); inline;
        begin
            outp[outNext1Line + outIndex] := interp8(c[3], c[1]);
        end;

        procedure pixel11_0(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := c[4];
        end;

        procedure pixel11_30(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := interp3(c[4], c[0]);
        end;

        procedure pixel11_31(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := interp3(c[4], c[3]);
        end;

        procedure pixel11_32(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := interp3(c[4], c[1]);
        end;

        procedure pixel11_70(); inline;
        begin
            outp[outNext1Line + outIndex + 1] := interp7(c[4], c[3], c[1]);
        end;

        procedure pixel12_0(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := c[4];
        end;

        procedure pixel12_30(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp3(c[4], c[2]);
        end;

        procedure pixel12_31(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp3(c[4], c[1]);
        end;

        procedure pixel12_32(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp3(c[4], c[5]);
        end;

        procedure pixel12_70(); inline;
        begin
            outp[outNext1Line + outIndex + 2] := interp7(c[4], c[5], c[1]);
        end;

        procedure pixel13_0(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := c[4];
        end;

        procedure pixel13_10(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp1(c[4], c[2]);
        end;

        procedure pixel13_12(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp1(c[4], c[5]);
        end;

        procedure pixel13_14(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp1(c[5], c[4]);
        end;

        procedure pixel13_21(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp2(c[5], c[4], c[1]);
        end;

        procedure pixel13_31(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp3(c[4], c[1]);
        end;

        procedure pixel13_50(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp5(c[5], c[4]);
        end;

        procedure pixel13_60(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp6(c[4], c[5], c[1]);
        end;

        procedure pixel13_61(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp6(c[4], c[5], c[2]);
        end;

        procedure pixel13_82(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp8(c[4], c[5]);
        end;

        procedure pixel13_83(); inline;
        begin
            outp[outNext1Line + outIndex + 3] := interp8(c[5], c[1]);
        end;

        procedure pixel20_0(); inline;
        begin
            outp[outNext2Line + outIndex] := c[4];
        end;

        procedure pixel20_10(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[4], c[6]);
        end;

        procedure pixel20_12(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel20_14(); inline;
        begin
            outp[outNext2Line + outIndex] := interp1(c[3], c[4]);
        end;

        procedure pixel20_21(); inline;
        begin
            outp[outNext2Line + outIndex] := interp2(c[3], c[4], c[7]);
        end;

        procedure pixel20_31(); inline;
        begin
            outp[outNext2Line + outIndex] := interp3(c[4], c[7]);
        end;

        procedure pixel20_50(); inline;
        begin
            outp[outNext2Line + outIndex] := interp5(c[3], c[4]);
        end;

        procedure pixel20_60(); inline;
        begin
            outp[outNext2Line + outIndex] := interp6(c[4], c[3], c[7]);
        end;

        procedure pixel20_61(); inline;
        begin
            outp[outNext2Line + outIndex] := interp6(c[4], c[3], c[6]);
        end;

        procedure pixel20_82(); inline;
        begin
            outp[outNext2Line + outIndex] := interp8(c[4], c[3]);
        end;

        procedure pixel20_83(); inline;
        begin
            outp[outNext2Line + outIndex] := interp8(c[3], c[7]);
        end;

        procedure pixel21_0(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := c[4];
        end;

        procedure pixel21_30(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp3(c[4], c[6]);
        end;

        procedure pixel21_31(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp3(c[4], c[7]);
        end;

        procedure pixel21_32(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp3(c[4], c[3]);
        end;

        procedure pixel21_70(); inline;
        begin
            outp[outNext2Line + outIndex + 1] := interp7(c[4], c[3], c[7]);
        end;

        procedure pixel22_0(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := c[4];
        end;

        procedure pixel22_30(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp3(c[4], c[8]);
        end;

        procedure pixel22_31(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp3(c[4], c[5]);
        end;

        procedure pixel22_32(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp3(c[4], c[7]);
        end;

        procedure pixel22_70(); inline;
        begin
            outp[outNext2Line + outIndex + 2] := interp7(c[4], c[5], c[7]);
        end;

        procedure pixel23_0(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := c[4];
        end;

        procedure pixel23_10(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp1(c[4], c[8]);
        end;

        procedure pixel23_11(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp1(c[4], c[5]);
        end;

        procedure pixel23_13(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp1(c[5], c[4]);
        end;

        procedure pixel23_21(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp2(c[5], c[4], c[7]);
        end;

        procedure pixel23_32(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp3(c[4], c[7]);
        end;

        procedure pixel23_50(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp5(c[5], c[4]);
        end;

        procedure pixel23_60(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp6(c[4], c[5], c[7]);
        end;

        procedure pixel23_61(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp6(c[4], c[5], c[8]);
        end;

        procedure pixel23_81(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp8(c[4], c[5]);
        end;

        procedure pixel23_83(); inline;
        begin
            outp[outNext2Line + outIndex + 3] := interp8(c[5], c[7]);
        end;

        procedure pixel30_0(); inline;
        begin
            outp[outNext3Line + outIndex] := c[4];
        end;

        procedure pixel30_11(); inline;
        begin
            outp[outNext3Line + outIndex] := interp1(c[4], c[7]);
        end;

        procedure pixel30_12(); inline;
        begin
            outp[outNext3Line + outIndex] := interp1(c[4], c[3]);
        end;

        procedure pixel30_20(); inline;
        begin
            outp[outNext3Line + outIndex] := interp2(c[4], c[7], c[3]);
        end;

        procedure pixel30_50(); inline;
        begin
            outp[outNext3Line + outIndex] := interp5(c[7], c[3]);
        end;

        procedure pixel30_80(); inline;
        begin
            outp[outNext3Line + outIndex] := interp8(c[4], c[6]);
        end;

        procedure pixel30_81(); inline;
        begin
            outp[outNext3Line + outIndex] := interp8(c[4], c[7]);
        end;

        procedure pixel30_82(); inline;
        begin
            outp[outNext3Line + outIndex] := interp8(c[4], c[3]);
        end;

        procedure pixel31_0(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := c[4];
        end;

        procedure pixel31_10(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp1(c[4], c[6]);
        end;

        procedure pixel31_11(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp1(c[4], c[7]);
        end;

        procedure pixel31_13(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp1(c[7], c[4]);
        end;

        procedure pixel31_21(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp2(c[7], c[4], c[3]);
        end;

        procedure pixel31_32(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp3(c[4], c[3]);
        end;

        procedure pixel31_50(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp5(c[7], c[4]);
        end;

        procedure pixel31_60(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp6(c[4], c[7], c[3]);
        end;

        procedure pixel31_61(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp6(c[4], c[7], c[6]);
        end;

        procedure pixel31_81(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp8(c[4], c[7]);
        end;

        procedure pixel31_83(); inline;
        begin
            outp[outNext3Line + outIndex + 1] := interp8(c[7], c[3]);
        end;

        procedure pixel32_0(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := c[4];
        end;

        procedure pixel32_10(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp1(c[4], c[8]);
        end;

        procedure pixel32_12(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp1(c[4], c[7]);
        end;

        procedure pixel32_14(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp1(c[7], c[4]);
        end;

        procedure pixel32_21(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp2(c[7], c[4], c[5]);
        end;

        procedure pixel32_31(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp3(c[4], c[5]);
        end;

        procedure pixel32_50(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp5(c[7], c[4]);
        end;

        procedure pixel32_60(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp6(c[4], c[7], c[5]);
        end;

        procedure pixel32_61(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp6(c[4], c[7], c[8]);
        end;

        procedure pixel32_82(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp8(c[4], c[7]);
        end;

        procedure pixel32_83(); inline;
        begin
            outp[outNext3Line + outIndex + 2] := interp8(c[7], c[5]);
        end;

        procedure pixel33_0(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := c[4];
        end;

        procedure pixel33_11(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp1(c[4], c[5]);
        end;

        procedure pixel33_12(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp1(c[4], c[7]);
        end;

        procedure pixel33_20(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp2(c[4], c[7], c[5]);
        end;

        procedure pixel33_50(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp5(c[7], c[5]);
        end;

        procedure pixel33_80(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp8(c[4], c[8]);
        end;

        procedure pixel33_81(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp8(c[4], c[5]);
        end;

        procedure pixel33_82(); inline;
        begin
            outp[outNext3Line + outIndex + 3] := interp8(c[4], c[7]);
        end;

    begin
        pixels := self.pixels16bit;
        width := self.width;
        height := self.height;
        transformTo16bit(self.inp, width, height, self.inScanline, pixels);
        w := square3x3w;
        c := square3x3c;
        pixCurrLine := 0;
        outCurrLine := 0;
        if turned then begin
            outCurrLine := outScanline * (height * 4 - 1);
            outScanline := -outScanline;
        end;
        for j := 0 to height - 1 do begin
            if j > 0 then begin
                pixPrevLine := pixCurrLine - width;
            end else begin
                pixPrevLine := pixCurrLine;
            end;
            if j < height - 1 then begin
                pixNextLine := pixCurrLine + width;
            end else begin
                pixNextLine := pixCurrLine;
            end;
            outNext1Line := outCurrLine + outScanline;
            outNext2Line := outNext1Line + outScanline;
            outNext3Line := outNext2Line + outScanline;
            outIndex := 0;
            for i := 0 to width - 1 do begin
                w[1] := pixels[pixPrevLine + i] and $ffff;
                w[4] := pixels[pixCurrLine + i] and $ffff;
                w[7] := pixels[pixNextLine + i] and $ffff;
                if i > 0 then begin
                    w[0] := pixels[pixPrevLine + i - 1] and $ffff;
                    w[3] := pixels[pixCurrLine + i - 1] and $ffff;
                    w[6] := pixels[pixNextLine + i - 1] and $ffff;
                end else begin
                    w[0] := w[1];
                    w[3] := w[4];
                    w[6] := w[7];
                end;
                if i < width - 1 then begin
                    w[2] := pixels[pixPrevLine + i + 1] and $ffff;
                    w[5] := pixels[pixCurrLine + i + 1] and $ffff;
                    w[8] := pixels[pixNextLine + i + 1] and $ffff;
                end else begin
                    w[2] := w[1];
                    w[5] := w[4];
                    w[8] := w[7];
                end;
                {
                    p:
                    +---+---+---+
                    | 0 | 1 | 2 |
                    +---+---+---+
                    | 3 | 4 | 5 |
                    +---+---+---+
                    | 6 | 7 | 8 |
                    +---+---+---+
                }
                { Собственно сам алгоритм }
                pattern := 0;
                if diff(w[4], w[0]) <> 0 then begin
                    pattern := pattern or $0001;
                end;
                if diff(w[4], w[1]) <> 0 then begin
                    pattern := pattern or $0002;
                end;
                if diff(w[4], w[2]) <> 0 then begin
                    pattern := pattern or $0004;
                end;
                if diff(w[4], w[3]) <> 0 then begin
                    pattern := pattern or $0008;
                end;
                if diff(w[4], w[5]) <> 0 then begin
                    pattern := pattern or $0010;
                end;
                if diff(w[4], w[6]) <> 0 then begin
                    pattern := pattern or $0020;
                end;
                if diff(w[4], w[7]) <> 0 then begin
                    pattern := pattern or $0040;
                end;
                if diff(w[4], w[8]) <> 0 then begin
                    pattern := pattern or $0080;
                end;
                for k := 0 to 8 do begin
                    c[k] := transformTo32bit(w[k]);
                end;
                case pattern of
                0, 1, 4, 5, 32, 33, 36, 37, 128, 129, 132, 133, 160, 161, 164, 165: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                2, 34, 130, 162: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                3, 35, 131, 163: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_61();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                6, 38, 134, 166: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_61();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                7, 39, 135: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                8, 12, 136, 140: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                9, 13, 137, 141: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel10_32();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                10, 138: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                        pixel11_0();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_61();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                11, 139: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_61();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                14, 142: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel02_32();
                        pixel03_82();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_50();
                        pixel01_83();
                        pixel02_13();
                        pixel03_11();
                        pixel10_21();
                        pixel11_70();
                    end;
                    pixel12_32();
                    pixel13_82();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                15, 143: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel02_32();
                        pixel03_82();
                        pixel10_0();
                        pixel11_0();
                    end else begin
                        pixel00_50();
                        pixel01_83();
                        pixel02_13();
                        pixel03_11();
                        pixel10_21();
                        pixel11_70();
                    end;
                    pixel12_32();
                    pixel13_82();
                    pixel20_10();
                    pixel21_30();
                    pixel22_70();
                    pixel23_60();
                    pixel30_80();
                    pixel31_61();
                    pixel32_60();
                    pixel33_20();
                end;
                16, 17, 48, 49: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                18, 50: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel12_0();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                19, 51: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_81();
                        pixel01_31();
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel00_12();
                        pixel01_14();
                        pixel02_83();
                        pixel03_50();
                        pixel12_70();
                        pixel13_21();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                20, 21, 52, 53: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    pixel03_81();
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel13_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                22, 54: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel12_0();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                23, 55: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_81();
                        pixel01_31();
                        pixel02_0();
                        pixel03_0();
                        pixel12_0();
                        pixel13_0();
                    end else begin
                        pixel00_12();
                        pixel01_14();
                        pixel02_83();
                        pixel03_50();
                        pixel12_70();
                        pixel13_21();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_30();
                    pixel23_10();
                    pixel30_20();
                    pixel31_60();
                    pixel32_61();
                    pixel33_80();
                end;
                24: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                25: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                26, 31: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel11_0();
                    pixel12_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                27: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                28: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                29: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                30: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel12_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_61();
                    pixel32_61();
                    pixel33_80();
                end;
                40, 44, 168, 172: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    pixel20_31();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel30_81();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                41, 45, 169: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel10_32();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel20_31();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel30_81();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                42, 170: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                        pixel20_31();
                        pixel30_81();
                    end else begin
                        pixel00_50();
                        pixel01_21();
                        pixel10_83();
                        pixel11_70();
                        pixel20_14();
                        pixel30_12();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_61();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                43, 171: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                        pixel11_0();
                        pixel20_31();
                        pixel30_81();
                    end else begin
                        pixel00_50();
                        pixel01_21();
                        pixel10_83();
                        pixel11_70();
                        pixel20_14();
                        pixel30_12();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_61();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                46, 174: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    pixel02_32();
                    pixel03_82();
                    pixel12_32();
                    pixel13_82();
                    pixel20_31();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel30_81();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                47, 175: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    pixel02_32();
                    pixel03_82();
                    pixel10_0();
                    pixel11_0();
                    pixel12_32();
                    pixel13_82();
                    pixel20_31();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel30_81();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                56: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                57: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                58: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                59: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel11_0();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                60: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                61: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                62: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel12_0();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                63: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_0();
                    pixel11_0();
                    pixel12_0();
                    pixel20_31();
                    pixel21_31();
                    pixel22_30();
                    pixel23_10();
                    pixel30_81();
                    pixel31_81();
                    pixel32_61();
                    pixel33_80();
                end;
                64, 65, 68, 69: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                66: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                67: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_61();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                70: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_61();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                71: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                72, 76: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_50();
                        pixel21_0();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                73, 77: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_82();
                        pixel10_32();
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel00_11();
                        pixel10_13();
                        pixel20_83();
                        pixel21_70();
                        pixel30_50();
                        pixel31_21();
                    end;
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                74, 107: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_61();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                75: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_61();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_61();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                78: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    pixel02_32();
                    pixel03_82();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                79: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_32();
                    pixel03_82();
                    pixel11_0();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                80, 81: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_61();
                    pixel21_30();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                82, 214: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel12_0();
                    pixel20_61();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                83: begin
                    pixel00_81();
                    pixel01_31();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_61();
                    pixel21_30();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                84, 85: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel03_81();
                        pixel13_31();
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel03_12();
                        pixel13_14();
                        pixel22_70();
                        pixel23_83();
                        pixel32_21();
                        pixel33_50();
                    end;
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel20_61();
                    pixel21_30();
                    pixel30_80();
                    pixel31_10();
                end;
                86: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel12_0();
                    pixel20_61();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                87: begin
                    pixel00_81();
                    pixel01_31();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel12_0();
                    pixel20_61();
                    pixel21_30();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                88, 248: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                end;
                89: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                90: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                91: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel11_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                92: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                93: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                94: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel12_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                95: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel11_0();
                    pixel12_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_30();
                    pixel23_10();
                    pixel30_80();
                    pixel31_10();
                    pixel32_10();
                    pixel33_80();
                end;
                96, 97, 100, 101: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_61();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                98: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_61();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                99: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_61();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_61();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                102: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_61();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_61();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                103: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_61();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                104, 108: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                105, 109: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_82();
                        pixel10_32();
                        pixel20_0();
                        pixel21_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel00_11();
                        pixel10_13();
                        pixel20_83();
                        pixel21_70();
                        pixel30_50();
                        pixel31_21();
                    end;
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                106: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                110: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_10();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                111: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    pixel02_32();
                    pixel03_82();
                    pixel10_0();
                    pixel11_0();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_61();
                    pixel32_10();
                    pixel33_80();
                end;
                112, 113: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel30_82();
                        pixel31_32();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_70();
                        pixel23_21();
                        pixel30_11();
                        pixel31_13();
                        pixel32_83();
                        pixel33_50();
                    end;
                end;
                114: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                    pixel30_82();
                    pixel31_32();
                end;
                115: begin
                    pixel00_81();
                    pixel01_31();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                    pixel30_82();
                    pixel31_32();
                end;
                116, 117: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    pixel03_81();
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel13_31();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                    pixel30_82();
                    pixel31_32();
                end;
                118: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel12_0();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_10();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                119: begin
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel00_81();
                        pixel01_31();
                        pixel02_0();
                        pixel03_0();
                        pixel12_0();
                        pixel13_0();
                    end else begin
                        pixel00_12();
                        pixel01_14();
                        pixel02_83();
                        pixel03_50();
                        pixel12_70();
                        pixel13_21();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_82();
                    pixel21_32();
                    pixel22_30();
                    pixel23_10();
                    pixel30_82();
                    pixel31_32();
                    pixel32_10();
                    pixel33_80();
                end;
                120: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                121: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                122: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_30();
                        pixel23_10();
                        pixel32_10();
                        pixel33_80();
                    end else begin
                        pixel22_0();
                        pixel23_11();
                        pixel32_12();
                        pixel33_20();
                    end;
                end;
                123: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                124: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                125: begin
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel00_82();
                        pixel10_32();
                        pixel20_0();
                        pixel21_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel00_11();
                        pixel10_13();
                        pixel20_83();
                        pixel21_70();
                        pixel30_50();
                        pixel31_21();
                    end;
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                126: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel12_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                127: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_0();
                    pixel11_0();
                    pixel12_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_30();
                    pixel23_10();
                    pixel32_10();
                    pixel33_80();
                end;
                144, 145, 176, 177: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel23_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                    pixel33_82();
                end;
                146, 178: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                        pixel23_32();
                        pixel33_82();
                    end else begin
                        pixel02_21();
                        pixel03_50();
                        pixel12_70();
                        pixel13_83();
                        pixel23_13();
                        pixel33_11();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                end;
                147, 179: begin
                    pixel00_81();
                    pixel01_31();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel23_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                    pixel33_82();
                end;
                148, 149, 180: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    pixel03_81();
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel13_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel23_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                    pixel33_82();
                end;
                150, 182: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel12_0();
                        pixel13_0();
                        pixel23_32();
                        pixel33_82();
                    end else begin
                        pixel02_21();
                        pixel03_50();
                        pixel12_70();
                        pixel13_83();
                        pixel23_13();
                        pixel33_11();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                end;
                151, 183: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel12_0();
                    pixel13_0();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel23_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                    pixel33_82();
                end;
                152: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                153: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                154: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                155: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                156: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                157: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                158: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel12_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                159: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel11_0();
                    pixel12_0();
                    pixel13_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_32();
                    pixel23_32();
                    pixel30_80();
                    pixel31_61();
                    pixel32_82();
                    pixel33_82();
                end;
                167: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_60();
                    pixel21_70();
                    pixel22_70();
                    pixel23_60();
                    pixel30_20();
                    pixel31_60();
                    pixel32_60();
                    pixel33_20();
                end;
                173: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel10_32();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel20_31();
                    pixel21_31();
                    pixel22_70();
                    pixel23_60();
                    pixel30_81();
                    pixel31_81();
                    pixel32_60();
                    pixel33_20();
                end;
                181: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    pixel03_81();
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel13_31();
                    pixel20_60();
                    pixel21_70();
                    pixel22_32();
                    pixel23_32();
                    pixel30_20();
                    pixel31_60();
                    pixel32_82();
                    pixel33_82();
                end;
                184: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                185: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                186: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                187: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                        pixel11_0();
                        pixel20_31();
                        pixel30_81();
                    end else begin
                        pixel00_50();
                        pixel01_21();
                        pixel10_83();
                        pixel11_70();
                        pixel20_14();
                        pixel30_12();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_10();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                188: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                189: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                190: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel12_0();
                        pixel13_0();
                        pixel23_32();
                        pixel33_82();
                    end else begin
                        pixel02_21();
                        pixel03_50();
                        pixel12_70();
                        pixel13_83();
                        pixel23_13();
                        pixel33_11();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                end;
                191: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel10_0();
                    pixel11_0();
                    pixel12_0();
                    pixel13_0();
                    pixel20_31();
                    pixel21_31();
                    pixel22_32();
                    pixel23_32();
                    pixel30_81();
                    pixel31_81();
                    pixel32_82();
                    pixel33_82();
                end;
                192, 193, 196, 197: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_61();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                194: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    pixel20_61();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                195: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_61();
                    pixel20_61();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                198: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_61();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    pixel20_61();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                199: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_61();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                200, 204: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                        pixel32_31();
                        pixel33_81();
                    end else begin
                        pixel20_21();
                        pixel21_70();
                        pixel30_50();
                        pixel31_83();
                        pixel32_14();
                        pixel33_12();
                    end;
                    pixel22_31();
                    pixel23_81();
                end;
                201, 205: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel10_32();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_31();
                    pixel23_81();
                    pixel32_31();
                    pixel33_81();
                end;
                202: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_61();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_31();
                    pixel23_81();
                    pixel32_31();
                    pixel33_81();
                end;
                203: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_61();
                    pixel20_10();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                206: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    pixel02_32();
                    pixel03_82();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_31();
                    pixel23_81();
                    pixel32_31();
                    pixel33_81();
                end;
                207: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel02_32();
                        pixel03_82();
                        pixel10_0();
                        pixel11_0();
                    end else begin
                        pixel00_50();
                        pixel01_83();
                        pixel02_13();
                        pixel03_11();
                        pixel10_21();
                        pixel11_70();
                    end;
                    pixel12_32();
                    pixel13_82();
                    pixel20_10();
                    pixel21_30();
                    pixel22_31();
                    pixel23_81();
                    pixel30_80();
                    pixel31_10();
                    pixel32_31();
                    pixel33_81();
                end;
                208, 209: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_61();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                210: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_61();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                211: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_10();
                    pixel20_61();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                212, 213: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel03_81();
                        pixel13_31();
                        pixel22_0();
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel03_12();
                        pixel13_14();
                        pixel22_70();
                        pixel23_83();
                        pixel32_21();
                        pixel33_50();
                    end;
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel20_61();
                    pixel21_30();
                    pixel30_80();
                    pixel31_10();
                end;
                215: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel12_0();
                    pixel13_0();
                    pixel20_61();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                216: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_61();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                217: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                218: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                end;
                219: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_10();
                    pixel20_10();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                220: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_10();
                        pixel21_30();
                        pixel30_80();
                        pixel31_10();
                    end else begin
                        pixel20_12();
                        pixel21_0();
                        pixel30_20();
                        pixel31_11();
                    end;
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                end;
                221: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel03_81();
                        pixel13_31();
                        pixel22_0();
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel03_12();
                        pixel13_14();
                        pixel22_70();
                        pixel23_83();
                        pixel32_21();
                        pixel33_50();
                    end;
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel20_10();
                    pixel21_30();
                    pixel30_80();
                    pixel31_10();
                end;
                222: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel12_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                223: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel11_0();
                    pixel12_0();
                    pixel13_0();
                    pixel20_10();
                    pixel21_30();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_80();
                    pixel31_10();
                end;
                224, 225, 228: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                226: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_61();
                    pixel11_30();
                    pixel12_30();
                    pixel13_61();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                227: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_61();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                229: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_60();
                    pixel03_20();
                    pixel10_60();
                    pixel11_70();
                    pixel12_70();
                    pixel13_60();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                230: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_61();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                231: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_32();
                    pixel03_82();
                    pixel10_81();
                    pixel11_31();
                    pixel12_32();
                    pixel13_82();
                    pixel20_82();
                    pixel21_32();
                    pixel22_31();
                    pixel23_81();
                    pixel30_82();
                    pixel31_32();
                    pixel32_31();
                    pixel33_81();
                end;
                232, 236: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_60();
                    pixel03_20();
                    pixel10_10();
                    pixel11_30();
                    pixel12_70();
                    pixel13_60();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel21_0();
                        pixel30_0();
                        pixel31_0();
                        pixel32_31();
                        pixel33_81();
                    end else begin
                        pixel20_21();
                        pixel21_70();
                        pixel30_50();
                        pixel31_83();
                        pixel32_14();
                        pixel33_12();
                    end;
                    pixel22_31();
                    pixel23_81();
                end;
                233, 237: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_60();
                    pixel03_20();
                    pixel10_32();
                    pixel11_32();
                    pixel12_70();
                    pixel13_60();
                    pixel20_0();
                    pixel21_0();
                    pixel22_31();
                    pixel23_81();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                    pixel32_31();
                    pixel33_81();
                end;
                234: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_80();
                        pixel01_10();
                        pixel10_10();
                        pixel11_30();
                    end else begin
                        pixel00_20();
                        pixel01_12();
                        pixel10_11();
                        pixel11_0();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel12_30();
                    pixel13_61();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_31();
                    pixel23_81();
                    pixel32_31();
                    pixel33_81();
                end;
                235: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_61();
                    pixel20_0();
                    pixel21_0();
                    pixel22_31();
                    pixel23_81();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                    pixel32_31();
                    pixel33_81();
                end;
                238: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_32();
                    pixel03_82();
                    pixel10_10();
                    pixel11_30();
                    pixel12_32();
                    pixel13_82();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel21_0();
                        pixel30_0();
                        pixel31_0();
                        pixel32_31();
                        pixel33_81();
                    end else begin
                        pixel20_21();
                        pixel21_70();
                        pixel30_50();
                        pixel31_83();
                        pixel32_14();
                        pixel33_12();
                    end;
                    pixel22_31();
                    pixel23_81();
                end;
                239: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    pixel02_32();
                    pixel03_82();
                    pixel10_0();
                    pixel11_0();
                    pixel12_32();
                    pixel13_82();
                    pixel20_0();
                    pixel21_0();
                    pixel22_31();
                    pixel23_81();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                    pixel32_31();
                    pixel33_81();
                end;
                240, 241: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_61();
                    pixel03_80();
                    pixel10_60();
                    pixel11_70();
                    pixel12_30();
                    pixel13_10();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_0();
                        pixel23_0();
                        pixel30_82();
                        pixel31_32();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel22_70();
                        pixel23_21();
                        pixel30_11();
                        pixel31_13();
                        pixel32_83();
                        pixel33_50();
                    end;
                end;
                242: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_10();
                        pixel03_80();
                        pixel12_30();
                        pixel13_10();
                    end else begin
                        pixel02_11();
                        pixel03_20();
                        pixel12_0();
                        pixel13_12();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel20_82();
                    pixel21_32();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    pixel30_82();
                    pixel31_32();
                end;
                243: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_10();
                    pixel03_80();
                    pixel10_81();
                    pixel11_31();
                    pixel12_30();
                    pixel13_10();
                    pixel20_82();
                    pixel21_32();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel22_0();
                        pixel23_0();
                        pixel30_82();
                        pixel31_32();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel22_70();
                        pixel23_21();
                        pixel30_11();
                        pixel31_13();
                        pixel32_83();
                        pixel33_50();
                    end;
                end;
                244, 245: begin
                    pixel00_20();
                    pixel01_60();
                    pixel02_81();
                    pixel03_81();
                    pixel10_60();
                    pixel11_70();
                    pixel12_31();
                    pixel13_31();
                    pixel20_82();
                    pixel21_32();
                    pixel22_0();
                    pixel23_0();
                    pixel30_82();
                    pixel31_32();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                246: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_61();
                    pixel11_30();
                    pixel12_0();
                    pixel20_82();
                    pixel21_32();
                    pixel22_0();
                    pixel23_0();
                    pixel30_82();
                    pixel31_32();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                247: begin
                    pixel00_81();
                    pixel01_31();
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel10_81();
                    pixel11_31();
                    pixel12_0();
                    pixel13_0();
                    pixel20_82();
                    pixel21_32();
                    pixel22_0();
                    pixel23_0();
                    pixel30_82();
                    pixel31_32();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                249: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_61();
                    pixel03_80();
                    pixel10_32();
                    pixel11_32();
                    pixel12_30();
                    pixel13_10();
                    pixel20_0();
                    pixel21_0();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                end;
                250: begin
                    pixel00_80();
                    pixel01_10();
                    pixel02_10();
                    pixel03_80();
                    pixel10_10();
                    pixel11_30();
                    pixel12_30();
                    pixel13_10();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                end;
                251: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                        pixel01_0();
                        pixel10_0();
                    end else begin
                        pixel00_50();
                        pixel01_50();
                        pixel10_50();
                    end;
                    pixel02_10();
                    pixel03_80();
                    pixel11_0();
                    pixel12_30();
                    pixel13_10();
                    pixel20_0();
                    pixel21_0();
                    pixel22_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel23_0();
                        pixel32_0();
                        pixel33_0();
                    end else begin
                        pixel23_50();
                        pixel32_50();
                        pixel33_50();
                    end;
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                end;
                252: begin
                    pixel00_80();
                    pixel01_61();
                    pixel02_81();
                    pixel03_81();
                    pixel10_10();
                    pixel11_30();
                    pixel12_31();
                    pixel13_31();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_0();
                    pixel23_0();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                253: begin
                    pixel00_82();
                    pixel01_82();
                    pixel02_81();
                    pixel03_81();
                    pixel10_32();
                    pixel11_32();
                    pixel12_31();
                    pixel13_31();
                    pixel20_0();
                    pixel21_0();
                    pixel22_0();
                    pixel23_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                254: begin
                    pixel00_80();
                    pixel01_10();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel02_0();
                        pixel03_0();
                        pixel13_0();
                    end else begin
                        pixel02_50();
                        pixel03_50();
                        pixel13_50();
                    end;
                    pixel10_10();
                    pixel11_30();
                    pixel12_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel20_0();
                        pixel30_0();
                        pixel31_0();
                    end else begin
                        pixel20_50();
                        pixel30_50();
                        pixel31_50();
                    end;
                    pixel21_0();
                    pixel22_0();
                    pixel23_0();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                255: begin
                    if diff(w[3], w[1]) <> 0 then begin
                        pixel00_0();
                    end else begin
                        pixel00_20();
                    end;
                    pixel01_0();
                    pixel02_0();
                    if diff(w[1], w[5]) <> 0 then begin
                        pixel03_0();
                    end else begin
                        pixel03_20();
                    end;
                    pixel10_0();
                    pixel11_0();
                    pixel12_0();
                    pixel13_0();
                    pixel20_0();
                    pixel21_0();
                    pixel22_0();
                    pixel23_0();
                    if diff(w[7], w[3]) <> 0 then begin
                        pixel30_0();
                    end else begin
                        pixel30_20();
                    end;
                    pixel31_0();
                    pixel32_0();
                    if diff(w[5], w[7]) <> 0 then begin
                        pixel33_0();
                    end else begin
                        pixel33_20();
                    end;
                end;
                end;
                { Конец алгоритма }
                inc(outIndex, 4);
            end;
            inc(pixCurrLine, width);
            inc(outCurrLine, outScanline * 4);
        end;
        returnToFPU();
    end;

    function Algorithm.getSourceWidth(): int;
    begin
        result := width;
    end;

    function Algorithm.getSourceHeight(): int;
    begin
        result := height;
    end;
{%endregion}

initialization {%region}
    init();
{%endregion}

end.

