/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package java.util;

import malik.emulator.util.*;

public class Vector extends Object
{
    private class Enumerator extends Object implements Enumeration
    {
        private int index;

        public Enumerator() {
        }

        public boolean hasMoreElements() {
            return index < (Vector.this).elementCount;
        }

        public Object nextElement() {
            int error = 0;
            Object result;
            Vector parent;
            synchronized((parent = Vector.this).monitor)
            {
                label0:
                {
                    int i;
                    if((i = index) >= parent.elementCount)
                    {
                        error = 1;
                        result = null;
                        break label0;
                    }
                    index = i + 1;
                    result = parent.elementData[i];
                }
            }
            if(error == 1)
            {
                throw new NoSuchElementException("Enumeration.nextElement: элементов больше не осталось.");
            }
            return result;
        }
    }

    protected int capacityIncrement;
    protected int elementCount;
    protected Object[] elementData;
    final Object monitor;

    public Vector() {
        this(16, 0);
    }

    public Vector(int initialCapacity) {
        this(initialCapacity, 0);
    }

    public Vector(int initialCapacity, int incrementCapacity) {
        if(initialCapacity < 0)
        {
            throw new IllegalArgumentException("Vector: аргумент initialCapacity не может быть отрицательным.");
        }
        this.capacityIncrement = incrementCapacity;
        this.elementData = new Object[initialCapacity];
        this.monitor = new Object();
    }

    public String toString() {
        StringBuilder result = (new StringBuilder()).append('[');
        synchronized(monitor)
        {
            Enumerator e = this.new Enumerator();
            for(int i = elementCount; i-- > 0; )
            {
                result.append(e.nextElement());
                if(i > 0) result.append(",\u0020");
            }
        }
        return result.append(']').toString();
    }

    public void trimToSize() {
        synchronized(monitor)
        {
            int c;
            Object[] d;
            if((c = elementCount) < (d = elementData).length) Array.copy(d, 0, elementData = new Object[c], 0, c);
        }
    }

    public void ensureCapacity(int minimumCapacity) {
        synchronized(monitor)
        {
            if(minimumCapacity > elementData.length) madeLarger(minimumCapacity);
        }
    }

    public void insertElementAt(Object element, int index) {
        int error = 0;
        synchronized(monitor)
        {
            label0:
            {
                int c;
                Object[] d;
                if(index > (c = elementCount) || index < 0)
                {
                    error = 1;
                    break label0;
                }
                if(c == (d = elementData).length) d = madeLarger(c + 1);
                if(c > index) Array.copy(d, index, d, index + 1, c - index);
                d[index] = element;
                elementCount = c + 1;
            }
        }
        if(error == 1)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.insertElementAt: аргумент index выходит из диапазона.");
        }
    }

    public void addElement(Object element) {
        synchronized(monitor)
        {
            int c;
            Object[] d;
            if((c = elementCount) == (d = elementData).length) d = madeLarger(c + 1);
            d[c] = element;
            elementCount = c + 1;
        }
    }

    public void removeElementAt(int index) {
        int error = 0;
        synchronized(monitor)
        {
            label0:
            {
                int c;
                Object[] d;
                if(index > (c = elementCount - 1) || index < 0)
                {
                    error = 1;
                    break label0;
                }
                d = elementData;
                if(c > index) Array.copy(d, index + 1, d, index, c - index);
                d[elementCount = c] = null;
            }
        }
        if(error == 1)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.removeElementAt: аргумент index выходит из диапазона.");
        }
    }

    public void removeAllElements() {
        synchronized(monitor)
        {
            Array.fill(elementData, 0, elementCount, null);
            elementCount = 0;
        }
    }

    public void copyInto(Object[] dst) {
        if(dst == null) return;
        synchronized(monitor)
        {
            Object[] d = elementData;
            for(int len = Math.min(elementCount, dst.length), i = 0; i < len; i++) dst[i] = d[i];
        }
    }

    public void setSize(int size) {
        if(size < 0)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.setSize: аргумент size не может быть отрицательным.");
        }
        synchronized(monitor)
        {
            int c;
            Object[] d;
            if(size > (d = elementData).length) d = madeLarger(size);
            if(size > (c = elementCount))
            {
                Array.fill(d, c, size - c, null);
            }
            else if(size < c)
            {
                Array.fill(d, size, c - size, null);
            }
            elementCount = size;
        }
    }

    public void setElementAt(Object element, int index) {
        int error = 0;
        synchronized(monitor)
        {
            label0:
            {
                if(index >= elementCount || index < 0)
                {
                    error = 1;
                    break label0;
                }
                elementData[index] = element;
            }
        }
        if(error == 1)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.setElementAt: аргумент index выходит из диапазона.");
        }
    }

    public boolean isEmpty() {
        return elementCount <= 0;
    }

    public boolean contains(Object element) {
        boolean result;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                Object[] d = elementData;
                if(element == null)
                {
                    result = Array.findb(d, elementCount - 1, null) >= 0;
                    break label0;
                }
                for(i = elementCount; i-- > 0; )
                {
                    if(!element.equals(d[i])) continue;
                    result = true;
                    break label0;
                }
                result = false;
            }
        }
        return result;
    }

    public boolean removeElement(Object element) {
        boolean result;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                int index;
                int c = elementCount;
                Object[] d = elementData;
                label1:
                {
                    if(element == null)
                    {
                        if((i = Array.findf(d, 0, null)) < c)
                        {
                            result = true;
                            break label1;
                        }
                    } else
                    {
                        for(i = 0; i < c; i++) if(element.equals(d[i]))
                        {
                            result = true;
                            break label1;
                        }
                    }
                    result = false;
                    break label0;
                }
                if((i = --c - (index = i)) > 0) Array.copy(d, index + 1, d, index, i);
                d[elementCount = c] = null;
            }
        }
        return result;
    }

    public int size() {
        return elementCount;
    }

    public int capacity() {
        return elementData.length;
    }

    public int indexOf(Object element) {
        int result;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                int c;
                Object[] d;
                if((c = elementCount) <= 0)
                {
                    result = -1;
                    break label0;
                }
                d = elementData;
                if(element == null)
                {
                    if((i = Array.findf(d, 0, null)) < c)
                    {
                        result = i;
                        break label0;
                    }
                } else
                {
                    for(i = 0; i < c; i++) if(element.equals(d[i]))
                    {
                        result = i;
                        break label0;
                    }
                }
                result = -1;
            }
        }
        return result;
    }

    public int indexOf(Object element, int startFromIndex) {
        int result;
        if(startFromIndex < 0) startFromIndex = 0;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                int c;
                Object[] d;
                if(startFromIndex >= (c = elementCount))
                {
                    result = -1;
                    break label0;
                }
                d = elementData;
                if(element == null)
                {
                    if((i = Array.findf(d, startFromIndex, null)) < c)
                    {
                        result = i;
                        break label0;
                    }
                } else
                {
                    for(i = startFromIndex; i < c; i++) if(element.equals(d[i]))
                    {
                        result = i;
                        break label0;
                    }
                }
                result = -1;
            }
        }
        return result;
    }

    public int lastIndexOf(Object element) {
        int result;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                int c;
                Object[] d;
                if((c = elementCount) <= 0)
                {
                    result = -1;
                    break label0;
                }
                d = elementData;
                if(element == null)
                {
                    if((i = Array.findb(d, c - 1, null)) >= 0)
                    {
                        result = i;
                        break label0;
                    }
                } else
                {
                    for(i = c; i-- > 0; ) if(element.equals(d[i]))
                    {
                        result = i;
                        break label0;
                    }
                }
                result = -1;
            }
        }
        return result;
    }

    public int lastIndexOf(Object element, int startFromIndex) {
        int result;
        int error = 0;
        synchronized(monitor)
        {
            label0:
            {
                int i;
                Object[] d;
                if((startFromIndex++) >= elementCount)
                {
                    error = 1;
                    result = 0;
                    break label0;
                }
                d = elementData;
                if(element == null)
                {
                    if((i = Array.findb(d, startFromIndex - 1, null)) >= 0)
                    {
                        result = i;
                        break label0;
                    }
                } else
                {
                    for(i = startFromIndex; i-- > 0; ) if(element.equals(d[i]))
                    {
                        result = i;
                        break label0;
                    }
                }
                result = -1;
            }
        }
        if(error == 1)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.lastIndexOf: аргумент startFromIndex выходит из диапазона.");
        }
        return result;
    }

    public Object firstElement() {
        int error = 0;
        Object result;
        synchronized(monitor)
        {
            label0:
            {
                if(elementCount <= 0)
                {
                    error = 1;
                    result = null;
                    break label0;
                }
                result = elementData[0];
            }
        }
        if(error == 1)
        {
            throw new NoSuchElementException("Vector.firstElement: вектор не содержит элементов.");
        }
        return result;
    }

    public Object lastElement() {
        int error = 0;
        Object result;
        synchronized(monitor)
        {
            label0:
            {
                int c;
                if((c = elementCount) <= 0)
                {
                    error = 1;
                    result = null;
                    break label0;
                }
                result = elementData[c - 1];
            }
        }
        if(error == 1)
        {
            throw new NoSuchElementException("Vector.lastElement: вектор не содержит элементов.");
        }
        return result;
    }

    public Object elementAt(int index) {
        int error = 0;
        Object result;
        synchronized(monitor)
        {
            label0:
            {
                if(index >= elementCount || index < 0)
                {
                    error = 1;
                    result = null;
                    break label0;
                }
                result = elementData[index];
            }
        }
        if(error == 1)
        {
            throw new ArrayIndexOutOfBoundsException("Vector.elementAt: аргумент index выходит из диапазона.");
        }
        return result;
    }

    public Enumeration elements() {
        return this.new Enumerator();
    }

    private Object[] madeLarger(int minimumCapacity) {
        int oc;
        int nc;
        int delta;
        Object[] result;
        oc = (result = elementData).length;
        if((nc = (delta = capacityIncrement) <= 0 ? (oc << 1) + 1 : oc + delta) < minimumCapacity) nc = minimumCapacity;
        Array.copy(result, 0, result = elementData = new Object[nc], 0, elementCount);
        return result;
    }
}
