/*
    Реализация спецификаций CLDC версии 1.1 (JSR-139), MIDP версии 2.1 (JSR-118)
    и других спецификаций для функционирования компактных приложений на языке
    Java (мидлетов) в среде программного обеспечения Малик Эмулятор.

    Copyright © 2016–2017, 2019–2022 Малик Разработчик

    Это свободная программа: вы можете перераспространять ее и/или изменять
    ее на условиях Меньшей Стандартной общественной лицензии GNU в том виде,
    в каком она была опубликована Фондом свободного программного обеспечения;
    либо версии 3 лицензии, либо (по вашему выбору) любой более поздней версии.

    Эта программа распространяется в надежде, что она будет полезной,
    но БЕЗО ВСЯКИХ ГАРАНТИЙ; даже без неявной гарантии ТОВАРНОГО ВИДА
    или ПРИГОДНОСТИ ДЛЯ ОПРЕДЕЛЕННЫХ ЦЕЛЕЙ. Подробнее см. в Меньшей Стандартной
    общественной лицензии GNU.

    Вы должны были получить копию Меньшей Стандартной общественной лицензии GNU
    вместе с этой программой. Если это не так, см.
    <https://www.gnu.org/licenses/>.
*/

package javax.microedition.lcdui;

import malik.emulator.application.*;
import malik.emulator.media.graphics.*;
import malik.emulator.microedition.lcdui.*;

public abstract class SurfaceScreen extends Screen
{
    private static final int INNER_MARGIN = 5; /* поля внутри экрана */
    private static final int OUTER_MARGIN = 8; /* отступы слева и справа от краёв экрана */
    private static final int WHEEL_SCROLL = 16; /* скорость прокрутки с помощью колеса мыши */

    class Helper extends Screen.Helper
    {
        private int scrollPos;

        public Helper() {
            this.scrollPos = -1;
        }

        protected void execute() {
            serviceSizeChanged();
            serviceScroll();
            servicePaint();
        }

        protected final void serviceScroll() {
            int pos;
            SurfaceScreen owner;
            if((owner = SurfaceScreen.this).getParentDisplay() == null) return;
            pos = owner.scroll.invalidatePosition();
            if(scrollPos == pos) return;
            try
            {
                owner.onClientScroll(scrollPos = pos, availableWidth(), availableHeight());
            }
            catch(RuntimeException e)
            {
                e.printRealStackTrace();
            }
        }

        protected final int position() {
            return scrollPos;
        }
    }

    private final class VertScrollBar extends CustomScrollBar
    {
        public VertScrollBar(boolean visibility, int range, Object monitor) {
            super(visibility, range, monitor);
        }

        public void repaint() {
            (SurfaceScreen.this).requestPaintAll();
        }

        public int getPage() {
            return (SurfaceScreen.this).getApplicationHeight();
        }

        protected void notifyFieldsChanged(int fields) {
            (SurfaceScreen.this).requestPaintAll();
        }
    }

    private int focused;
    private final int size;
    private final int[] rect;
    final CustomScrollBar scroll;
    private final ScrollBarStyle style;

    SurfaceScreen(String title, int verticalRange, boolean scrollbarVisibility, ScrollBarStyle style) {
        super(title, null, true);
        if(style == null) style = new ScreenScrollBarStyle();
        this.size = style.size();
        this.rect = new int[4];
        this.scroll = new VertScrollBar(scrollbarVisibility, verticalRange, null);
        this.style = style;
    }

    public final ScrollBar getScrollBar() {
        return scroll;
    }

    abstract void paint(ScreenGraphics render);

    void paintBackground(ScreenGraphics render, int width, int height, byte visibleElements, byte drawingElements) {
        int base;
        int rectLeft;
        int rectTop;
        int rectWidth;
        int rectHeight;
        int elementHeight;
        Display parent;
        Displayable active;
        if((parent = getParentDisplay()) != null && (active = parent.getActiveBackground()) != null)
        {
            Displayable.Helper helper;
            (helper = active.helper).justPaint();
            helper.execute();
            render.reset();
            render.setARGBColor(0xc0000000);
        } else
        {
            render.setARGBColor(0xff000000);
        }
        render.fillRect(0, 0, width, height);
        base = height - getPanelHeight();
        elementHeight = (visibleElements & PANEL) != 0 ? base : height;
        rectLeft = OUTER_MARGIN;
        rectTop = base >> 2;
        rectWidth = width - (OUTER_MARGIN << 1);
        rectHeight = elementHeight - ((base >> 2) << 1);
        render.setARGBColor(0x60000000);
        render.fillRect(rectLeft - 6, rectTop - 6, rectWidth + 12, rectHeight + 12);
        render.fillRect(rectLeft - 4, rectTop - 4, rectWidth + 8, rectHeight + 8);
        render.fillRect(rectLeft - 2, rectTop - 2, rectWidth + 4, rectHeight + 4);
        render.setColor(RasterCanvas.getSystemColor(0x04));
        render.fillRect(rectLeft, rectTop, rectWidth, rectHeight);
    }

    void paintClient(ScreenGraphics render, int width, int height, int clipLeft, int clipTop, int clipWidth, int clipHeight, Image clipBuffer) {
        boolean svisible;
        int base = getTotalHeight() - getPanelHeight();
        int spos;
        int size = this.size;
        int left = OUTER_MARGIN + INNER_MARGIN;
        int top = (base >> 2) + INNER_MARGIN;
        int right = left;
        int bottom = top;
        int tx = left + render.getTranslateX();
        int ty = top + render.getTranslateY();
        ScrollBar scroll = this.scroll;
        ScrollBarStyle style = this.style;
        svisible = scroll.visibility;
        spos = scroll.position;
        width -= left + right + (svisible ? size : 0);
        height -= top + bottom;
        if(svisible)
        {
            render.reset();
            render.translate(tx + width, ty);
            style.verticalScrollBarPaintEvent(scroll, render, size, height);
        }
        render.reset();
        render.restricts(tx, ty, width, height);
        render.setStartPoint(tx, ty - spos);
        render.setClip(tx, ty, width, height);
        render.clipRect(tx + clipLeft, ty + clipTop, clipWidth, clipHeight);
        render.translate(tx, ty - spos);
        paint(render);
    }

    void onPaint(ScreenGraphics render, byte drawingElements, int clipLeft, int clipTop, int clipWidth, int clipHeight, Image clipBuffer) {
        super.onPaint(render, (byte) (CLIENT | PANEL), clipLeft, clipTop, clipWidth, clipHeight, clipBuffer);
    }

    boolean onKeyboardEvent(KeyboardEvent event) {
        if(!super.onKeyboardEvent(event)) onClientKeyboardEvent(event);
        return true;
    }

    boolean onPointerEvent(PointerEvent event) {
        boolean down;
        int b;
        int f;
        if(super.onPointerEvent(event)) return true;
        switch(event.getAction())
        {
        case PointerEvent.ACTION_POINTER_PRESSED:
            if((down = (b = event.getButton()) == PointerEvent.BUTTON_WHEEL_DOWN) || b == PointerEvent.BUTTON_WHEEL_UP)
            {
                scroll.scroll(down ? WHEEL_SCROLL : -WHEEL_SCROLL);
                return true;
            }
            return handlePointerEvent(focused = getFocusedElement(event.getX(), event.getY(), b), event);
        case PointerEvent.ACTION_BUTTON_PRESSED:
            if((down = (b = event.getButton()) == PointerEvent.BUTTON_WHEEL_DOWN) || b == PointerEvent.BUTTON_WHEEL_UP)
            {
                scroll.scroll(down ? WHEEL_SCROLL : -WHEEL_SCROLL);
                return true;
            }
            /* fall through */
        case PointerEvent.ACTION_POINTER_DRAGGED:
            return handlePointerEvent(focused, event);
        case PointerEvent.ACTION_BUTTON_RELEASED:
            return (b = event.getButton()) == PointerEvent.BUTTON_WHEEL_DOWN || b == PointerEvent.BUTTON_WHEEL_UP || handlePointerEvent(focused, event);
        case PointerEvent.ACTION_POINTER_RELEASED:
            f = focused;
            focused = 0;
            return (b = event.getButton()) == PointerEvent.BUTTON_WHEEL_DOWN || b == PointerEvent.BUTTON_WHEEL_UP || handlePointerEvent(f, event);
        }
        return false;
    }

    Displayable.Helper createHelper() {
        return this.new Helper();
    }

    void onClientScroll(int position, int clientWidth, int clientHeight) {
    }

    final void setFullScreenMode(boolean fullScreen) {
    }

    final boolean isFullScreenMode() {
        return false;
    }

    final int getMarginLeft() {
        return OUTER_MARGIN + INNER_MARGIN;
    }

    final int getMarginTop() {
        return ((getTotalHeight() - getPanelHeight()) >> 2) + INNER_MARGIN;
    }

    final int getMarginRight() {
        return (OUTER_MARGIN + INNER_MARGIN) + (scroll.visibility ? size : 0);
    }

    final int getMarginBottom() {
        return ((getTotalHeight() - getPanelHeight()) >> 2) + INNER_MARGIN;
    }

    private boolean handlePointerEvent(int focused, PointerEvent event) {
        int a;
        int[] r = rect;
        switch(focused)
        {
        case 2:
            event.translate(r[0], r[1]);
            style.verticalScrollBarPointerEvent(scroll, event, r[2], r[3]);
            return true;
        case 3:
            event.translate(r[0], r[1] - scroll.position);
            onClientPointerEvent(event);
            return true;
        case 4:
            if((a = event.getAction()) == PointerEvent.ACTION_POINTER_DRAGGED || a == PointerEvent.ACTION_BUTTON_RELEASED || a == PointerEvent.ACTION_POINTER_RELEASED)
            {
                event.translate(r[0], r[1]);
                scroll.scroll(event.historicalY(1) - event.getY());
            }
            return true;
        }
        return false;
    }

    private int getFocusedElement(int x, int y, int button) {
        boolean svisible = scroll.visibility;
        byte e = getElements();
        int height;
        int base = (height = getTotalHeight()) - getPanelHeight();
        int l = OUTER_MARGIN + INNER_MARGIN;
        int t = (base >> 2) + INNER_MARGIN;
        int w = getTotalWidth() - ((OUTER_MARGIN + INNER_MARGIN) << 1);
        int h = height - (t << 1) - ((e & PANEL) != 0 ? height - base : 0);
        int s = size;
        int[] r = rect;
        if(svisible)
        {
            int sl = l + w - s;
            if(x >= sl && x < sl + s && y >= t && y < t + h)
            {
                r[0] = sl;
                r[1] = t;
                r[2] = s;
                r[3] = h;
                return 2;
            }
        }
        if(x >= l && x < l + w && y >= t && y < t + h)
        {
            r[0] = l;
            r[1] = t;
            r[2] = w;
            r[3] = h;
            return button == PointerEvent.BUTTON_WHEEL ? 4 : 3;
        }
        return 0;
    }
}
